/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;


/**
 * Represents a date - year, month, day in the range Jan 1, 1000 - Dec 31, 9999,
 * with a lot of utilities.
 * <p>
 * The framework is aware of leap years, and checks all dates for validity.
 * <p>
 * Date objects are always immutable - they cannot be changed once created.
 * <p>
 * The natural order of this class orders instances in chronological order, and
 * two instances are considered equal if they represent the same year, month
 * and day.
 * <p>
 * <b>Example usage</b>
 * <table>
 * <tr valign="top">
 * <td>Create new instances:</td>
 * <td>
 * <code>DayDate.factory.now()</code><br>
 * <code>DayDate.factory.getDefault()</code><br>
 * <code>DayDate.factory.parse(...)</code>
 * </td>
 * </tr>
 * <tr valign="top">
 * <td>Formatting:</td>
 * <td>
 * <code>DayDate.iso().formatXXX()</code><br>
 * <code>DayDate.eur().formatXXX()</code><br>
 * <code>DayDate.us().formatXXX()</code>
 * </td>
 * </tr>
 * <tr valign="top">
 * <td>Serializing</td>
 * <td>
 * <code>DayDate.serialize()</code><br>
 * <code>DayDate.factory.deserialize()</code>
 * </td>
 * </tr>
 * </table>
 * 
 * @author Andreas Enblom
 */
public interface DayDate extends Comparable<DayDate>, java.io.Serializable {

    /**
     * A default implementaton of {@link DayDateFactory}, for creating new
     * instances. The factory will create {@link java.io.Serializable
     * serializable} instances.
     */
    public static final DayDateFactory factory = new DayDateFactoryImpl();

    /**
     * @return The year represented by this date in the range 1000-9999.
     */
    int year();

    /**
     * @return The {@link Month month} represented by this date.
     */
    Month month();

    /**
     * @return The day of month represented by this date in the range 1-28,
     *         1-29, 1-30 or 1-31, depending on what month and year it is.
     *         Leap-years are handled correctly.
     */
    int day();

    /**
     * Calculates the day of week of this date.
     * 
     * @return The day of week.
     */
    DayOfWeek getDayOfWeek();

    /**
     * Adds or subtracts the given number of years to this date. If the
     * resulting date is out of range, a runtime exception is thrown.
     * <p>
     * In the rare case that this is Feb 29 on a leap year, and adding the given
     * offset results in a non leap year, the date will be set to Feb 28.
     * 
     * @param offset The offset in years. Can be positive, zero or negative.
     * @return The resulting date.
     * @throws IllegalArgumentException If the resulting date is out of range.
     */
    DayDate plusYears(int offset);

    /**
     * Adds or subtracts the given number of months to this date. If the
     * resulting date is out of range, a runtime exception is thrown.
     * <p>
     * If adding the given number of months would make the day of month over the
     * new month's maximum it will be reduced to the proper maximum. Hence,
     * adding three months to 2011-05-31 will give 2011-08-31, but adding four
     * will give 2011-09-30. And adding 13 months to 2010-01-30 will give the
     * date 2011-02-28.
     * 
     * @param offset The offset in months. Can be positive, zero or negative.
     * @return The resulting date.
     * @throws IllegalArgumentException If the resulting date is out of range.
     */
    DayDate plusMonths(int offset);

    /**
     * Adds or subtracts the given number of days to this date. If the
     * resulting date is out of range, a runtime exception is thrown.
     * 
     * @param offset The offset in days. Can be positive, zero or negative.
     * @return The resulting time.
     * @throws IllegalArgumentException If the resulting date is out of range.
     */
    DayDate plusDays(int offset);

    /**
     * Determines whether this is in a later year than the given date.
     * 
     * @param date The date to compare to.
     * @return Whether this is in a later year than the given date.
     */
    boolean isLaterYearThan(DayDate date);

    /**
     * Determines whether this is in the same year as the given date.
     * 
     * @param date The date to compare to.
     * @return Whether this is in the same year as the given date.
     */
    boolean isSameYearAs(DayDate date);

    /**
     * Determines whether this is in a later month than the given date. A month
     * of a later year is always considered to be later, e.g. Jan, 2011 is a
     * later month than Nov, 2010.
     * 
     * @param date The date to compare to.
     * @return Whether this is in a later month than the given date.
     */
    boolean isLaterMonthThan(DayDate date);

    /**
     * Determines whether this is in the same month and the same year as the
     * given date.
     * 
     * @param date The date to compare to.
     * @return Whether this is in the same month as the given date.
     */
    boolean isSameMonthAs(DayDate date);

    @Override
    int hashCode();

    @Override
    boolean equals(Object other);

    /**
     * Provides a formatter for this date instance that formats day dates
     * according to the ISO-8601 standard. This means that dates are formatted
     * as <code>YYYY-MM-DD</code> and <code>YYYYMMDD</code>.
     * <p>
     * See also {@link #eur()} and {@link #us()}.
     * 
     * @return A formatter for this date that uses the ISO-8601 standard.
     */
    DayDateFormatter iso();

    /**
     * Provides a formatter for this date instance that formats day dates
     * according to some de-facto European standard. This means that dates are
     * formatted as <code>DD.MM.YYYY</code> and <code>DDMMYYYY</code>.
     * <p>
     * See also {@link #iso()} and {@link #us()}.
     * 
     * @return A formatter for this date that uses some de-facto European
     *         standard.
     */
    DayDateFormatter eur();

    /**
     * Provides a formatter for this date instance that formats day dates
     * according to the US standard. This means that dates are formatted as
     * <code>MM/DD/YYYY</code> and <code>MMDDYYYY</code>.
     * <p>
     * See also {@link #iso()} and {@link #eur()}.
     * 
     * @return A formatter for this date that uses the US standard.
     */
    DayDateFormatter us();

    /**
     * @return The date on the format <code>YYYYMMDD</code>.
     */
    String serialize();

    /**
     * Overrides {@link Object#toString()}.
     * 
     * @return The date on the format <code>YYYY-MM-DD</code>
     */
    @Override
    String toString();

    /**
     * Creates an int representation of the date, which when written in base
     * 10 is the string <code>YYYYMMDD</code>. E.g, the date 2011-03-02 is
     * represented as the integer 20110302.
     * 
     * @return The int representation.
     */
    int toInt();

}
