/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Default implementation of {@link DayDateFactory} that produces
 * {@link DayDateImpl} instances.
 * 
 * @author Andreas Enblom
 */
class DayDateFactoryImpl implements DayDateFactory {

    private static final java.util.TimeZone UTC = java.util.TimeZone.getTimeZone("UTC");
    private static final DayDate defaultDayDate = new DayDateImpl((short) 1970, (byte) 1, (byte) 1);

    @Override
    public DayDate now() {
        java.util.Calendar calendar = java.util.Calendar.getInstance();
        return convert(calendar);
    }

    @Override
    public DayDate utc() {
        java.util.Calendar calendar = java.util.Calendar.getInstance(UTC);
        return convert(calendar);
    }

    @Override
    public DayDate getDefault() {
        return defaultDayDate;
    }

    @Override
    public DayDate deserialize(String date) throws IllegalArgumentException {
        if (date == null || date.length() != 8) {
            throw new IllegalArgumentException("The serialization string " + date + " is not correct");
        }

        // YYYY MM DD
        // 0123 45 67
        String year   = date.substring(0, 4);
        String month  = date.substring(4, 6);
        String day    = date.substring(6, 8);

        DayDate deserialized = parse(year, month, day);
        if (deserialized == null) {
            throw new IllegalArgumentException("The serialization string " + date + " is not correct");
        }

        return deserialized;
    }

    @Override
    public DayDate parse(String year, String month, String day) {
        try {
            return parse(
                    Integer.parseInt(year),
                    Integer.parseInt(month),
                    Integer.parseInt(day));
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public DayDate parse(int year, int month, int day) {
        if (year < 1000 || year > 9999) {
            return null;
        }
        if (month < 1 || month > 12) {
            return null;
        }
        if (day < 1 || day > DayDateImpl.getMaxDay((short) year, (byte) month)) {
            return null;
        }
        return new DayDateImpl((short) year, (byte) month, (byte) day);
    }

    @Override
    public DayDate parse(String date) {
        String year;
        String month;
        String day;

        if (date.length() == 10 && date.charAt(4) == '-' && date.charAt(7) == '-') {
            // Format: YYYY-MM-DD
            //         0123456789
            year  = date.substring(0, 4);
            month = date.substring(5, 7);
            day   = date.substring(8);

        } else if (date.length() == 8) {
            // Format: YYYYMMDD
            //         01234567
            year  = date.substring(0, 4);
            month = date.substring(4, 6);
            day   = date.substring(6);

        } else {
            // No supported format
            return null;
        }

        return parse(year, month, day);
    }

    @Override
    public DayDate fromInt(int date) throws IllegalArgumentException {
        return deserialize(Integer.toString(date));
    }

    private static DayDate convert(java.util.Calendar calendar) {
        // Check the range of the year; the other field will not be out of range
        // as we trust java.util.Calendar on this point.
        int yearInt = calendar.get(java.util.Calendar.YEAR);
        DayDateImpl.checkYearRange(yearInt);

        short year = (short) yearInt;

        byte month;
        switch (calendar.get(java.util.Calendar.MONTH)) {
        case java.util.Calendar.JANUARY:
            month = 1;
            break;
        case java.util.Calendar.FEBRUARY:
            month = 2;
            break;
        case java.util.Calendar.MARCH:
            month = 3;
            break;
        case java.util.Calendar.APRIL:
            month = 4;
            break;
        case java.util.Calendar.MAY:
            month = 5;
            break;
        case java.util.Calendar.JUNE:
            month = 6;
            break;
        case java.util.Calendar.JULY:
            month = 7;
            break;
        case java.util.Calendar.AUGUST:
            month = 8;
            break;
        case java.util.Calendar.SEPTEMBER:
            month = 9;
            break;
        case java.util.Calendar.OCTOBER:
            month = 10;
            break;
        case java.util.Calendar.NOVEMBER:
            month = 11;
            break;
        case java.util.Calendar.DECEMBER:
            month = 12;
            break;
        default:
            throw new IllegalStateException("Unknown month from java.util.Calendar: " + calendar.get(java.util.Calendar.MONTH));
        }

        byte day = (byte) calendar.get(java.util.Calendar.DAY_OF_MONTH);

        return new DayDateImpl(year, month, day);
    }

}
