/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Provides methods for formatting a {@link DayDate} object as a human-readable
 * string.
 * <p>
 * The following notation will be used:
 * <table>
 * </tr><td><code>YYYY</code>:</td><td>The year written with four digits, e.g. 1066, 1969, 2011</td></tr>
 * </tr><td><code>YY</code>:</td><td>The year written with two digits, e.g. 66, 69, 11</td></tr>
 * </tr><td><code>MM</code>:</td><td>The month written with two digits, e.g. 02 for February and 10 for October</td></tr>
 * </tr><td><code>DD</code>:</td><td>The day of the month written with two digits, e.g. 02, 15, 31</td></tr>
 * </table>
 * <p>
 * There are three standard formatters: ISO, EUR and US, provided by
 * {@link Time#iso()}, {@link Time#eur()} and {@link Time#us()}, respectively.
 * The ISO formatter formats timestamps according to the ISO-8601 standard, the
 * EUR formatter formats timestamps according to some de-facto European
 * standard, and the US formatter formats timestamps according to US standards.
 * The main difference between how these these formatters format dates is
 * summarized here:
 * <table>
 * <td>ISO:</td><td><code>YYYY-MM-DD</code>, <code>YYYYMMDD</code></td></tr>
 * <td>EUR:</td><td><code>DD.MM.YYYY</code>, <code>DDMMYYYY</code></td></tr>
 * <td>US:</td><td><code>MM/DD/YYYY</code>, <code>MMDDYYYY</code></td></tr>
 * </table>
 * 
 * @author Andreas Enblom
 */
public interface DayDateFormatter {

    /**
     * Formats the date as a compact string, with no delimiters between year,
     * month and date. The year is written with four digits.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYYYMMDD</code></td></tr>
     * <tr><td>EUR</td><td><code>DDMMYYYY</code></td></tr>
     * <tr><td>US</td><td><code>MMDDYYYY</code></td></tr>
     * </table>
     * 
     * @return The date, formatted as a string.
     */
    String formatCompactDate();

    /**
     * Formats the date as a compact string, with no delimiters between year,
     * month and date. The year is written with two digits.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYMMDD</code></td></tr>
     * <tr><td>EUR</td><td><code>DDMMYY</code></td></tr>
     * <tr><td>US</td><td><code>MMDDYY</code></td></tr>
     * </table>
     * 
     * @return The date, formatted as a string.
     */
    String formatCompactShortDate();

    /**
     * Formats the date as a string, with delimiters between year, month and
     * day. The year is written with four digits.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYYY-MM-DD</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YYYY</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YYYY</code></td></tr>
     * </table>
     * 
     * @return The date, formatted as a string.
     */
    String formatDate();

    /**
     * Formats the date in a shorter way. The year is written with two digits.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YY-MM-DD</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YY</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YY</code></td></tr>
     * </table>
     * 
     * @return The date, formatted as a string.
     */
    String formatShortDate();

}
