/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Provides methods for formatting a {@link Time} object as a human-readable
 * string.
 * <p>
 * The following notation will be used:
 * <table>
 * </tr><td><code>YYYY</code>:</td><td>The year written with four digits, e.g. 1066, 1969, 2011</td></tr>
 * </tr><td><code>YY</code>:</td><td>The year written with two digits, e.g. 66, 69, 11</td></tr>
 * </tr><td><code>MM</code>:</td><td>The month written with two digits, e.g. 02 for February and 10 for October</td></tr>
 * </tr><td><code>DD</code>:</td><td>The day of the month written with two digits, e.g. 02, 15, 31</td></tr>
 * </tr><td><code>hh</code>:</td><td>The hour of the time written with two digits. A 24-hour scale is used, so this will be in the range 00-23.</td></tr>
 * </tr><td><code>mm</code>:</td><td>The minute of the time written with two digits; in the range 00-59.</td></tr>
 * </tr><td><code>ss</code>:</td><td>The second of the time written with two digits; in the range 00-59.</td></tr>
 * </tr><td><code>nnn</code>:</td><td>The millisecond of the time written with three digits; in the range 000-999.</td></tr>
 * </table>
 * <p>
 * There are three standard formatters: ISO, EUR and US, provided by
 * {@link Time#iso()}, {@link Time#eur()} and {@link Time#us()}, respectively.
 * The ISO formatter formats timestamps according to the ISO-8601 standard, the
 * EUR formatter formats timestamps according to some de-facto European
 * standard, and the US formatter formats timestamps according to US standards.
 * The main difference between these formatters is how dates are formatted:
 * <table>
 * <td>ISO:</td><td><code>YYYY-MM-DD</code>, <code>YYYYMMDD</code></td></tr>
 * <td>EUR:</td><td><code>DD.MM.YYYY</code>, <code>DDMMYYYY</code></td></tr>
 * <td>US:</td><td><code>MM/DD/YYYY</code>, <code>MMDDYYYY</code></td></tr>
 * </table>
 * 
 * @author Andreas Enblom
 */
public interface TimeFormatter {

    /**
     * Formats the date of this time as a string, with delimiters between year,
     * month and day. The year is written with four digits.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYYY-MM-DD</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YYYY</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YYYY</code></td></tr>
     * </table>
     * 
     * @return The date of this time, formatted as a string.
     */
    String formatDate();

    /**
     * Formats the date of this time as a string, with delimiters between year,
     * month and day. The year is written with two digits.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YY-MM-DD</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YY</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YY</code></td></tr>
     * </table>
     * 
     * @return The date of the time, formatted as a string.
     */
    String formatShortDate();

    /**
     * Formats the date of the time as a compact string, with no delimiters
     * between year, month and date. The year is written with four digits.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYYYMMDD</code></td></tr>
     * <tr><td>EUR</td><td><code>DDMMYYYY</code></td></tr>
     * <tr><td>US</td><td><code>MMDDYYYY</code></td></tr>
     * </table>
     * 
     * @return The date of this time, formatted as a string.
     */
    String formatCompactDate();

    /**
     * Formats the date of the time as a compact string, with no delimiters
     * between year, month and date. The year is written with two digits.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYMMDD</code></td></tr>
     * <tr><td>EUR</td><td><code>DDMMYY</code></td></tr>
     * <tr><td>US</td><td><code>MMDDYY</code></td></tr>
     * </table>
     * 
     * @return The date of this time, formatted as a string.
     */
    String formatCompactShortDate();

    /**
     * Formats the time of day of the time as a string, with delimiters between
     * the different parts of this time. The formatting will include hour,
     * minute and second.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hh:mm:ss</code></td></tr>
     * <tr><td>EUR</td><td><code>hh:mm:ss</code></td></tr>
     * <tr><td>US</td><td><code>hh:mm:ss</code></td></tr>
     * </table>
     * 
     * @return The time of day of the time, formatted as a string.
     */
    String formatTime();

    /**
     * Formats the time of day of the time as a string, with delimiters between
     * the different parts of this time. The formatting will include hour,
     * minute, second and millisecond.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hh:mm:ss.nnn</code></td></tr>
     * <tr><td>EUR</td><td><code>hh:mm:ss.nnn</code></td></tr>
     * <tr><td>US</td><td><code>hh:mm:ss.nnn</code></td></tr>
     * </table>
     * 
     * @return The time of day of the time, formatted as a string.
     */
    String formatLongTime();

    /**
     * Formats the time of day of the time as a string, with delimiters between
     * the different parts of this time. The formatting will include hour and
     * minute.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hh:mm</code></td></tr>
     * <tr><td>EUR</td><td><code>hh:mm</code></td></tr>
     * <tr><td>US</td><td><code>hh:mm</code></td></tr>
     * </table>
     * 
     * @return The time of day of the time, formatted as a string.
     */
    String formatShortTime();

    /**
     * Formats the time of day of this time as a compact string, with no
     * delimiters between the different parts of this time. The formatting will
     * include hour, minute and second.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hhmmss</code></td></tr>
     * <tr><td>EUR</td><td><code>hhmmss</code></td></tr>
     * <tr><td>US</td><td><code>hhmmss</code></td></tr>
     * </table>
     * 
     * @return The time of day of this time, formatted as a string.
     */
    String formatCompactTime();

    /**
     * Formats the time of day of this time as a compact string, with no
     * delimiters between the different parts of this time. The formatting will
     * include hour, minute, second and millisecond.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hhmmssnnn</code></td></tr>
     * <tr><td>EUR</td><td><code>hhmmssnnn</code></td></tr>
     * <tr><td>US</td><td><code>hhmmssnnn</code></td></tr>
     * </table>
     * 
     * @return The time of day of this time, formatted as a string.
     */
    String formatCompactLongTime();

    /**
     * Formats the time of day of this time as a compact string, with no
     * delimiters between the different parts of this time. The formatting will
     * include hour and minute.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hhmm</code></td></tr>
     * <tr><td>EUR</td><td><code>hhmm</code></td></tr>
     * <tr><td>US</td><td><code>hhmm</code></td></tr>
     * </table>
     * 
     * @return The time of day of this time, formatted as a string.
     */
    String formatCompactShortTime();

    /**
     * Formats the the time as a full string. The year is written with four
     * digits, and the time of day will include hour, minute and second.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYYY-MM-DD hh:mm:ss</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YYYY hh:mm:ss</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YYYY hh:mm:ss</code></td></tr>
     * </table>
     * 
     * @return The time, formatted as an full string.
     */
    String formatDateAndTime();

    /**
     * Formats the the time as a full string. The year is written with four
     * digits, and the time of day will include hour, minute, second and
     * millisecond.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYYY-MM-DD hh:mm:ss.nnn</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YYYY hh:mm:ss.nnn</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YYYY hh:mm:ss.nnn</code></td></tr>
     * </table>
     * 
     * @return The time, formatted as an full string.
     */
    String formatDateAndLongTime();

    /**
     * Formats the the time as a full string. The year is written with four
     * digits, and the time of day will include hour and minute.
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYYY-MM-DD hh:mm</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YYYY hh:mm</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YYYY hh:mm</code></td></tr>
     * </table>
     * 
     * @return The time, formatted as an full string.
     */
    String formatDateAndShortTime();

    /**
     * Formats the the time as a full string. The year is written with two
     * digits, and the time of day will include hour, minute and second.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YY-MM-DD hh:mm:ss</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YY hh:mm:ss</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YY hh:mm:ss</code></td></tr>
     * </table>
     * 
     * @return The time, formatted as an full string.
     */
    String formatShortDateAndTime();

    /**
     * Formats the the time as a full string. The year is written with two
     * digits, and the time of day will include hour, minute, second and
     * millisecond.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YY-MM-DD hh:mm:ss.nnn</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YY hh:mm:ss.nnn</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YY hh:mm:ss.nnn</code></td></tr>
     * </table>
     * 
     * @return The time, formatted as an full string.
     */
    String formatShortDateAndLongTime();

    /**
     * Formats the the time as a full string. The year is written with two
     * digits, and the time of day will include hour and minute.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YY-MM-DD hh:mm</code></td></tr>
     * <tr><td>EUR</td><td><code>DD.MM.YY hh:mm</code></td></tr>
     * <tr><td>US</td><td><code>MM/DD/YY hh:mm</code></td></tr>
     * </table>
     * 
     * @return The time, formatted as an full string.
     */
    String formatShortDateAndShortTime();

    /**
     * Formats the the time as a full string. The date is formatted without
     * delimiters and with the year written with two digits. The time of day
     * will include hour, minute and second.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYMMDD hh:mm:ss</code></td></tr>
     * <tr><td>EUR</td><td><code>DDMMYY hh:mm:ss</code></td></tr>
     * <tr><td>US</td><td><code>MMDDYY hh:mm:ss</code></td></tr>
     * </table>
     * 
     * @return The time, formatted as an full string.
     */
    String formatCompactShortDateAndTime();

    /**
     * Formats the the time as a full string. The date is formatted without
     * delimiters and with the year written with two digits. The time of day
     * will include hour, minute, second and millisecond.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYMMDD hh:mm:ss.nnn</code></td></tr>
     * <tr><td>EUR</td><td><code>DDMMYY hh:mm:ss.nnn</code></td></tr>
     * <tr><td>US</td><td><code>MMDDYY hh:mm:ss.nnn</code></td></tr>
     * </table>
     * 
     * @return The time, formatted as an full string.
     */
    String formatCompactShortDateAndLongTime();

    /**
     * Formats the the time as a full string. The date is formatted without
     * delimiters and with the year written with two digits. The time of day
     * will include hour and minute.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>YYMMDD hh:mm</code></td></tr>
     * <tr><td>EUR</td><td><code>DDMMYY hh:mm</code></td></tr>
     * <tr><td>US</td><td><code>MMDDYY hh:mm</code></td></tr>
     * </table>
     * 
     * @return The time, formatted as an full string.
     */
    String formatCompactShortDateAndShortTime();

}
