/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Default implementation of {@link Time}, aiming at small memory usage and
 * efficient standard operations. Conversion to java.util.Date is expensive,
 * though.
 * 
 * @author Andreas Enblom
 */
class TimeImpl implements Time {

    /*
     * Implementation notice: Offset calculation uses long for minutes, seconds
     * and millis, and int for all others. This is because int is sufficient
     * for hour, day, month and year offsets in the range 1000-01-01 --
     * 9999-12-31.
     */

    final short year;   // 1000 - 9999
    final byte  month;  // 1 - 12
    final byte  day;    // 1 - 31
    final byte  hour;   // 0 - 23
    final byte  minute; // 0 - 59
    final byte  second; // 0 - 59
    final short millis; // 0 - 999

    /**
     * Creates a new instance with the given year, month, day hour, minute,
     * second and millisecond. No verification that the numbers are in the
     * correct range is done, or that the date is valid, is done.
     * 
     * @param year The year in the range 1000-9999.
     * @param month The month in the range 1-12.
     * @param day The day of month in the range 1-31.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     */
    TimeImpl(short year, byte month, byte day, byte hour, byte minute, byte second, short millis) {
        this.year = year;
        this.month = month;
        this.day = day;
        this.hour = hour;
        this.minute = minute;
        this.second = second;
        this.millis = millis;
    }

    @Override
    public DayDate getDate() {
        return new DayDateImpl(year, month, day);
    }

    @Override
    public TimeOfDay getTimeOfDay() {
        return new TimeOfDayImpl(hour, minute, second, millis);
    }

    @Override
    public int year() {
        return year;
    }

    @Override
    public Month month() {
        return Month.fromNum(month);
    }

    @Override
    public int day() {
        return day;
    }

    @Override
    public int hour() {
        return hour;
    }

    @Override
    public int minute() {
        return minute;
    }

    @Override
    public int second() {
        return second;
    }

    @Override
    public int millis() {
        return millis;
    }

    @Override
    public DayOfWeek getDayOfWeek() {
        return DayDateImpl.getDayOfWeek(year, month, day);
    }

    @Override
    public Time plusYears(int offset) {
        return new TimeImpl(
                DayDateImpl.addYearGetNewYear(year, offset),
                DayDateImpl.addYearGetNewMonth(year, month, offset),
                DayDateImpl.addYearGetNewDay(year, month, day, offset),
                hour,
                minute,
                second,
                millis);
    }

    @Override
    public Time plusMonths(int offset) {
        return new TimeImpl(
                DayDateImpl.addMonthGetNewYear(year, month, offset),
                DayDateImpl.addMonthGetNewMonth(month, offset),
                DayDateImpl.addMonthGetNewDay(year, month, day, offset),
                hour,
                minute,
                second,
                millis);
    }

    @Override
    public Time plusDays(int offset) {
        DayDateImpl newDate = DayDateImpl.plusDays(year, month, day, offset);
        return new TimeImpl(newDate.year, newDate.month, newDate.day, hour, minute, second, millis);
    }

    @Override
    public Time plusHours(int offset) {
        byte newHour = addHoursGetNewHour(hour, offset);
        int dateOffset = addHoursGetDayOffset(hour, offset);
        DayDateImpl newDate = DayDateImpl.plusDays(year, month, day, dateOffset);
        return new TimeImpl(newDate.year, newDate.month, newDate.day, newHour, minute, second, millis);
    }

    @Override
    public Time plusMinutes(long offset) {
        byte newMinute = addMinutesGetNewMinute(minute, offset);
        int hourOffset = addMinutesGetHourOffset(minute, offset);
        byte newHour = addHoursGetNewHour(hour, hourOffset);
        int dateOffset = addHoursGetDayOffset(hour, hourOffset);
        DayDateImpl newDate = DayDateImpl.plusDays(year, month, day, dateOffset);
        return new TimeImpl(newDate.year, newDate.month, newDate.day, newHour, newMinute, second, millis);
    }

    @Override
    public Time plusSeconds(long offset) {
        byte newSecond = addSecondsGetNewSecond(second, offset);
        long minuteOffset = addSecondsGetMinuteOffset(second, offset);
        byte newMinute = addMinutesGetNewMinute(minute, minuteOffset);
        int hourOffset = addMinutesGetHourOffset(minute, minuteOffset);
        byte newHour = addHoursGetNewHour(hour, hourOffset);
        int dateOffset = addHoursGetDayOffset(hour, hourOffset);
        DayDateImpl newDate = DayDateImpl.plusDays(year, month, day, dateOffset);
        return new TimeImpl(newDate.year, newDate.month, newDate.day, newHour, newMinute, newSecond, millis);
    }

    @Override
    public Time plusMillis(long offset) {
        short newMillis = addMillisGetNewMillis(millis, offset);
        long secondOffset = addMillisGetSecondOffset(millis, offset);
        byte newSecond = addSecondsGetNewSecond(second, secondOffset);
        long minuteOffset = addSecondsGetMinuteOffset(second, secondOffset);
        byte newMinute = addMinutesGetNewMinute(minute, minuteOffset);
        int hourOffset = addMinutesGetHourOffset(minute, minuteOffset);
        byte newHour = addHoursGetNewHour(hour, hourOffset);
        int dateOffset = addHoursGetDayOffset(hour, hourOffset);
        DayDateImpl newDate = DayDateImpl.plusDays(year, month, day, dateOffset);
        return new TimeImpl(newDate.year, newDate.month, newDate.day, newHour, newMinute, newSecond, newMillis);
    }

    @Override
    public boolean isAfter(Time time) {
        if (time == null) {
            throw new IllegalArgumentException("The time cannot be null");
        }
        return compareTo(time) >= 0;
    }

    @Override
    public boolean isAfter(TimeOfDay time) {
        if (time == null) {
            throw new IllegalArgumentException("The time cannot be null");
        }
        return getTimeOfDay().compareTo(time) >= 0;
    }

    @Override
    public boolean isAfter(int hours, int minutes) {
        if (hour > hours) {
            return true;
        } else if (hour == hours && minute >= minutes) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isAfter(int hours, int minutes, int seconds) {
        if (hour > hours) {
            return true;
        } else if (hour == hours && minute > minutes) {
            return true;
        } else if (hour == hours && minute == minutes && second >= seconds) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isBefore(Time time) {
        if (time == null) {
            throw new IllegalArgumentException("The time cannot be null");
        }
        return compareTo(time) < 0;
    }

    @Override
    public boolean isBefore(TimeOfDay time) {
        if (time == null) {
            throw new IllegalArgumentException("The time cannot be null");
        }
        return getTimeOfDay().compareTo(time) < 0;
    }

    @Override
    public boolean isBefore(int hours, int minutes) {
        if (hour < hours) {
            return true;
        } else if (hour == hours && minute < minutes) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isBefore(int hours, int minutes, int seconds) {
        if (hour < hours) {
            return true;
        } else if (hour == hours && minute < minutes) {
            return true;
        } else if (hour == hours && minute == minutes && second < seconds) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isLaterYearThan(Time time) {
        return year > time.year();
    }

    @Override
    public boolean isSameYearAs(Time time) {
        return year == time.year();
    }

    @Override
    public boolean isLaterMonthThan(Time time) {
        return
                (year > time.year()) ||
                (year == time.year() && month > time.month().toNum());
    }

    @Override
    public boolean isSameMonthAs(Time time) {
        return
                year == time.year() &&
                month == time.month().toNum();
    }

    @Override
    public boolean isLaterDayThan(Time time) {
        return
                (year > time.year()) ||
                (year == time.year() && month > time.month().toNum()) ||
                (year == time.year() && month == time.month().toNum() && day > time.day());
    }

    @Override
    public boolean isSameDayAs(Time time) {
        return
                year == time.year() &&
                month == time.month().toNum() &&
                day == time.day();
    }

    @Override
    public boolean isLaterHourThan(Time time) {
        return
                (year > time.year()) ||
                (year == time.year() && month > time.month().toNum()) ||
                (year == time.year() && month == time.month().toNum() && day > time.day()) ||
                (year == time.year() && month == time.month().toNum() && day == time.day() && hour > time.hour());
    }

    @Override
    public boolean isSameHourAs(Time time) {
        return
                year == time.year() &&
                month == time.month().toNum() &&
                day == time.day() &&
                hour == time.hour();
    }

    @Override
    public boolean isLaterMinuteThan(Time time) {
        return
                (year > time.year()) ||
                (year == time.year() && month > time.month().toNum()) ||
                (year == time.year() && month == time.month().toNum() && day > time.day()) ||
                (year == time.year() && month == time.month().toNum() && day == time.day() && hour > time.hour()) ||
                (year == time.year() && month == time.month().toNum() && day == time.day() && hour == time.hour() && minute > time.minute());
    }

    @Override
    public boolean isSameMinuteAs(Time time) {
        return
                year == time.year() &&
                month == time.month().toNum() &&
                day == time.day() &&
                hour == time.hour() &&
                minute == time.minute();
    }

    @Override
    public boolean isLaterSecondThan(Time time) {
        return
                (year > time.year()) ||
                (year == time.year() && month > time.month().toNum()) ||
                (year == time.year() && month == time.month().toNum() && day > time.day()) ||
                (year == time.year() && month == time.month().toNum() && day == time.day() && hour > time.hour()) ||
                (year == time.year() && month == time.month().toNum() && day == time.day() && hour == time.hour() && minute > time.minute()) ||
                (year == time.year() && month == time.month().toNum() && day == time.day() && hour == time.hour() && minute == time.minute() && second > time.second());
    }

    @Override
    public boolean isSameSecondAs(Time time) {
        return
                year == time.year() &&
                month == time.month().toNum() &&
                day == time.day() &&
                hour == time.hour() &&
                minute == time.minute() &&
                second == time.second();
    }

    @Override
    public int compareTo(Time other) {
        long thisVal = toLong();
        long otherVal = other.toLong();
        return (thisVal < otherVal ? -1 : (thisVal == otherVal ? 0 : 1));
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Time)) {
            return false;
        }
        return this.toLong() == ((Time) obj).toLong();
    }

    @Override
    public int hashCode() {
        return Long.valueOf(toLong()).hashCode();
    }

    @Override
    public TimeFormatter iso() {
        return new TimeFormatterIsoImpl(this);
    }

    @Override
    public TimeFormatter eur() {
        return new TimeFormatterEurImpl(this);
    }

    @Override
    public TimeFormatter us() {
        return new TimeFormatterUsImpl(this);
    }

    @Override
    public String serialize() {
        return
                year +
                (month < 10 ? "0" : "") + month +
                (day < 10 ? "0" : "") + day +
                (hour < 10 ? "0" : "") + hour +
                (minute < 10 ? "0" : "") + minute +
                (second < 10 ? "0" : "") + second +
                (millis < 100 ? (millis < 10 ? "00" : "0") : "") + millis;
    }

    @Override
    public String toString() {
        return iso().formatDateAndLongTime();
    }

    @Override
    public long toLong() {
        // yyyy mm dd hh mm ss nnn
        // 6543 21 09 87 65 43 210

        return
                //       01234567890123
                year   * 10000000000000L +
                //       012345678901
                month  * 100000000000L +
                //       0123456789
                day    * 1000000000L +
                //       01234567
                hour   * 10000000L +
                //       012345
                minute * 100000L +
                //       0123
                second * 1000L +
                //       0
                millis * 1L;
    }

    @Override
    public java.util.Date toJavaUtilDate() {
        return toJavaUtilDate(java.util.TimeZone.getDefault());
    }

    @Override
    public java.util.Date toJavaUtilDate(java.util.TimeZone timeZone) {
        int month;
        switch(month()) {
        case JANUARY:
            month = java.util.Calendar.JANUARY;
            break;
        case FEBRUARY:
            month = java.util.Calendar.FEBRUARY;
            break;
        case MARCH:
            month = java.util.Calendar.MARCH;
            break;
        case APRIL:
            month = java.util.Calendar.APRIL;
            break;
        case MAY:
            month = java.util.Calendar.MAY;
            break;
        case JUNE:
            month = java.util.Calendar.JUNE;
            break;
        case JULY:
            month = java.util.Calendar.JULY;
            break;
        case AUGUST:
            month = java.util.Calendar.AUGUST;
            break;
        case SEPTEMBER:
            month = java.util.Calendar.SEPTEMBER;
            break;
        case OCTOBER:
            month = java.util.Calendar.OCTOBER;
            break;
        case NOVEMBER:
            month = java.util.Calendar.NOVEMBER;
            break;
        case DECEMBER:
            month = java.util.Calendar.DECEMBER;
            break;
        default:
            throw new IllegalStateException("Unknwon month: " + month());
        }

        java.util.Calendar calendar = java.util.Calendar.getInstance(timeZone);
        calendar.set(java.util.Calendar.YEAR, year);
        calendar.set(java.util.Calendar.MONTH, month);
        calendar.set(java.util.Calendar.DAY_OF_MONTH, day);
        calendar.set(java.util.Calendar.HOUR_OF_DAY, hour);
        calendar.set(java.util.Calendar.MINUTE, minute);
        calendar.set(java.util.Calendar.SECOND, second);
        calendar.set(java.util.Calendar.MILLISECOND, millis);

        return calendar.getTime();
    }

    static int addHoursGetDayOffset(byte hour, int offset) {
        int dayOffset = offset / 24;
        int newHour = hour + (offset % 24);
        if (newHour < 0) {
            dayOffset--;
        } else if (newHour > 23) {
            dayOffset++;
        }
        return dayOffset;
    }

    static byte addHoursGetNewHour(byte hour, int offset) {
        // Complicated expression to handle negative offsets
        return (byte) ((((hour + offset) % 24) + 24) % 24);
    }

    static int addMinutesGetHourOffset(byte minute, long offset) {
        long hourOffset = offset / 60;
        int newMinute = (int) (minute + (offset % 60));
        if (newMinute < 0) {
            hourOffset--;
        } else if (newMinute > 59) {
            hourOffset++;
        }
        return (int) hourOffset;
    }

    static byte addMinutesGetNewMinute(byte minute, long offset) {
        // Complicated expression to handle negative offsets
        return (byte) ((((minute + offset) % 60) + 60) % 60);
    }

    static long addSecondsGetMinuteOffset(byte second, long offset) {
        long minuteOffset = offset / 60;
        int newSecond = (int) (second + (offset % 60));
        if (newSecond < 0) {
            minuteOffset--;
        } else if (newSecond > 59) {
            minuteOffset++;
        }
        return minuteOffset;
    }

    static byte addSecondsGetNewSecond(byte second, long offset) {
        // Complicated expression to handle negative offsets
        return (byte) ((((second + offset) % 60) + 60) % 60);
    }

    static long addMillisGetSecondOffset(short millis, long offset) {
        long secondOffset = offset / 1000;
        int newMillis = (int) (millis + (offset % 1000));
        if (newMillis < 0) {
            secondOffset--;
        } else if (newMillis > 999) {
            secondOffset++;
        }
        return secondOffset;
    }

    static short addMillisGetNewMillis(short millis, long offset) {
        // Complicated expression to handle negative offsets
        return (short) ((((millis + offset) % 1000) + 1000) % 1000);
    }

}
