package org.encalmo.models

import java.lang.management.ManagementFactory
import java.math.BigInteger
import java.nio.charset.StandardCharsets
import java.security.MessageDigest
import java.security.SecureRandom

opaque type CUID2 = String

object CUID2
    extends OpaqueStringWithPattern[
      CUID2,
      "[abcdefghijklmnopqrstuvwxyz0123456789]{24,32}"
    ]:

  // CUID configuration
  val LENGTH_STANDARD = 24

  inline def randomCUID2(): CUID2 = randomCUID2(LENGTH_STANDARD)

  def randomCUID2(length: Int): CUID2 = {
    synchronized {
      val time =
        java.lang.Long.toString(System.currentTimeMillis(), NUMBER_BASE)

      val firstLetter = ALPHABET_ARRAY(
        Math.abs(nextIntValue()) % ALPHABET_ARRAY.length
      )
      val hash = computeHash(
        time + createEntropy(length) + nextCounterValue() + MACHINE_FINGERPRINT,
        length
      )

      s"$firstLetter${hash.substring(1, length)}"
    }
  }

  private val NUMBER_BASE = 36

  private val ALPHABET_ARRAY = Array[Char](
    'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w',
    'x', 'y', 'z'
  )

  private val PRIME_NUMBER_ARRAY = Array[Int](
    109717, 109721, 109741, 109751, 109789, 109793, 109807, 109819, 109829, 109831
  )

  // Counter
  private var counter: Int = Integer.MAX_VALUE
  private inline def nextCounterValue(): Int = {
    counter =
      if (counter < Integer.MAX_VALUE)
      then counter
      else Math.abs(nextIntValue())

    counter
  }

  private val RANDOM_BUFFER_SIZE = 4096
  private val RANDOM_BUFFER = Array.ofDim[Byte](RANDOM_BUFFER_SIZE)
  private val NUMBER_GENERATOR = new SecureRandom()

  private lazy val MACHINE_FINGERPRINT: String = {
    val machineName = ManagementFactory.getRuntimeMXBean().getName()
    val machineNameTokenArray = machineName.split("@")
    val pid = machineNameTokenArray(0)
    val hostname = machineNameTokenArray(1)

    var acc = hostname.length() + NUMBER_BASE
    for (ch <- hostname) do acc = acc + ch

    val idBlock = padWithZero(pid, 2)
    val nameBlock = padWithZero(Integer.toString(acc), 2)

    idBlock + nameBlock
  }

  private var randomBufferIndex: Int = RANDOM_BUFFER_SIZE - 1

  private inline def nextRandomBufferIndex: Int = {
    randomBufferIndex = randomBufferIndex + 1
    randomBufferIndex
  }

  private inline def nextIntValue(): Int = {
    if (randomBufferIndex == RANDOM_BUFFER_SIZE - 1) {
      NUMBER_GENERATOR.nextBytes(RANDOM_BUFFER)
      randomBufferIndex = 0
    }

    (RANDOM_BUFFER(randomBufferIndex) << 24)
      | ((RANDOM_BUFFER(nextRandomBufferIndex) & 0xff) << 16)
      | ((RANDOM_BUFFER(nextRandomBufferIndex) & 0xff) << 8)
      | (RANDOM_BUFFER(nextRandomBufferIndex) & 0xff)
  }

  private inline def padWithZero(str: String, size: Int): String = {
    val paddedString = "000000000" + str
    paddedString.substring(paddedString.length() - size)
  }

  private inline def computeHash(content: String, saltLength: Int): String = {
    val salt = createEntropy(saltLength)
    new BigInteger(
      MessageDigest
        .getInstance("SHA3-256")
        .digest((content + salt).getBytes(StandardCharsets.UTF_8))
    )
      .toString(NUMBER_BASE)
  }

  private inline def createEntropy(length: Int): String = {

    var primeNumber: Int = 0

    val stringBuilder = new StringBuilder(length)

    while (stringBuilder.length() < length) {
      primeNumber = PRIME_NUMBER_ARRAY(
        Math.abs(nextIntValue()) % PRIME_NUMBER_ARRAY.length
      )
      stringBuilder.append(
        Integer.toString(primeNumber * nextIntValue(), NUMBER_BASE)
      )
    }

    stringBuilder.toString()
  }

end CUID2
