package org.encalmo.aws

import software.amazon.awssdk.core.SdkBytes
import software.amazon.awssdk.services.kms.model.*

import java.nio.charset.StandardCharsets
import java.util.Base64

/** Collection of AWS KMS service operations. */
object AwsKmsApi {

  /** Creates a digital signature for a message or message digest by using the private key in an asymmetric signing KMS
    * key. To verify the signature, use the Verify operation, or use the public key in the same asymmetric KMS key
    * outside of KMS.
    *
    * Digital signatures are generated and verified by using asymmetric key pair, such as an RSA or ECC pair that is
    * represented by an asymmetric KMS key. The key owner (or an authorized user) uses their private key to sign a
    * message. Anyone with the public key can verify that the message was signed with that particular private key and
    * that the message hasn't changed since it was signed.
    */
  inline def signRawMessage(
      keyId: String,
      algorithm: String,
      message: String
  )(using
      AwsClient
  ): String =
    AwsClient.invoke(s"signRawMessage") {
      val signature = summon[AwsClient].kms
        .sign(
          SignRequest
            .builder()
            .keyId(keyId)
            .signingAlgorithm(algorithm)
            .message(SdkBytes.fromUtf8String(message))
            .messageType(MessageType.RAW)
            .build()
        )
        .signature()
      String(
        Base64.getEncoder().encode(signature.asByteArrayUnsafe()),
        StandardCharsets.UTF_8
      )
    }

  /** Creates a digital signature for a message or message digest by using the private key in an asymmetric signing KMS
    * key. To verify the signature, use the Verify operation, or use the public key in the same asymmetric KMS key
    * outside of KMS.
    *
    * Digital signatures are generated and verified by using asymmetric key pair, such as an RSA or ECC pair that is
    * represented by an asymmetric KMS key. The key owner (or an authorized user) uses their private key to sign a
    * message. Anyone with the public key can verify that the message was signed with that particular private key and
    * that the message hasn't changed since it was signed.
    */
  inline def signRawMessage(
      keyId: String,
      algorithm: String,
      bytes: Array[Byte]
  )(using
      AwsClient
  ): Array[Byte] =
    AwsClient.invoke(s"signRawMessage") {
      summon[AwsClient].kms
        .sign(
          SignRequest
            .builder()
            .keyId(keyId)
            .signingAlgorithm(algorithm)
            .message(SdkBytes.fromByteArrayUnsafe(bytes))
            .messageType(MessageType.RAW)
            .build()
        )
        .signature()
        .asByteArrayUnsafe()
    }

  /** Verifies a digital signature that was generated by the Sign operation.
    *
    * Verification confirms that an authorized user signed the message with the specified KMS key and signing algorithm,
    * and the message hasn't changed since it was signed. If the signature is verified, the value of the SignatureValid
    * field in the response is True. If the signature verification fails, the Verify operation fails with an
    * KmsInvalidSignatureException exception.
    */
  inline def verifyRawMessage(
      keyId: String,
      algorithm: String,
      message: String,
      signatureBase64: String
  )(using
      AwsClient
  ): Boolean =
    AwsClient.invoke(s"verifyRawMessage") {
      try
        (summon[AwsClient].kms
          .verify(
            VerifyRequest
              .builder()
              .keyId(keyId)
              .signingAlgorithm(algorithm)
              .message(SdkBytes.fromUtf8String(message))
              .messageType(MessageType.RAW)
              .signature(
                SdkBytes
                  .fromByteArrayUnsafe(
                    Base64
                      .getDecoder()
                      .decode(signatureBase64.getBytes(StandardCharsets.UTF_8))
                  )
              )
              .build()
          )
          .signatureValid())
      catch { case e: KmsInvalidSignatureException => false }
    }

  /** Verifies a digital signature that was generated by the Sign operation.
    *
    * Verification confirms that an authorized user signed the message with the specified KMS key and signing algorithm,
    * and the message hasn't changed since it was signed. If the signature is verified, the value of the SignatureValid
    * field in the response is True. If the signature verification fails, the Verify operation fails with an
    * KmsInvalidSignatureException exception.
    */
  inline def verifyRawMessage(
      keyId: String,
      algorithm: String,
      bytes: Array[Byte],
      signature: Array[Byte]
  )(using
      AwsClient
  ): Boolean =
    AwsClient.invoke(s"verifyRawMessage") {
      try
        (summon[AwsClient].kms
          .verify(
            VerifyRequest
              .builder()
              .keyId(keyId)
              .signingAlgorithm(algorithm)
              .message(SdkBytes.fromByteArrayUnsafe(bytes))
              .messageType(MessageType.RAW)
              .signature(SdkBytes.fromByteArrayUnsafe(signature))
              .build()
          )
          .signatureValid())
      catch { case e: KmsInvalidSignatureException => false }
    }

  /** Encrypts plaintext of up to 4,096 bytes using a KMS key. You can use a symmetric or asymmetric KMS key with a
    * KeyUsage of ENCRYPT_DECRYPT.
    *
    * You can use this operation to encrypt small amounts of arbitrary data, such as a personal identifier or database
    * password, or other sensitive information. You don't need to use the Encrypt operation to encrypt a data key. The
    * GenerateDataKey and GenerateDataKeyPair operations return a plaintext data key and an encrypted copy of that data
    * key.
    *
    * When you use an asymmetric KMS key to encrypt or reencrypt data, be sure to record the KMS key and encryption
    * algorithm that you choose. You will be required to provide the same KMS key and encryption algorithm when you
    * decrypt the data. If the KMS key and algorithm do not match the values used to encrypt the data, the decrypt
    * operation fails.
    *
    * You are not required to supply the key ID and encryption algorithm when you decrypt with symmetric encryption KMS
    * keys because KMS stores this information in the ciphertext blob. KMS cannot store metadata in ciphertext generated
    * with asymmetric keys. The standard format for asymmetric key ciphertext does not include configurable fields.
    *
    * The maximum size of the data that you can encrypt varies with the type of KMS key and the encryption algorithm
    * that you choose.
    *
    * Symmetric encryption KMS keys
    *
    *   - SYMMETRIC_DEFAULT: 4096 bytes
    *
    *   - RSA_2048
    *     - RSAES_OAEP_SHA_1: 214 bytes
    *     - RSAES_OAEP_SHA_256: 190 bytes
    *   - RSA_3072
    *     - RSAES_OAEP_SHA_1: 342 bytes
    *     - RSAES_OAEP_SHA_256: 318 bytes
    *   - RSA_4096
    *     - RSAES_OAEP_SHA_1: 470 bytes
    *     - RSAES_OAEP_SHA_256: 446 bytes
    */
  inline def encryptMessage(
      keyId: String,
      algorithm: String,
      bytes: Array[Byte]
  )(using
      AwsClient
  ): Array[Byte] =
    AwsClient.invoke(s"encryptMessage") {
      summon[AwsClient].kms
        .encrypt(
          EncryptRequest
            .builder()
            .keyId(keyId)
            .encryptionAlgorithm(algorithm)
            .plaintext(SdkBytes.fromByteArrayUnsafe(bytes))
            .build()
        )
        .ciphertextBlob()
        .asByteArrayUnsafe()
    }

  /** Decrypts ciphertext that was encrypted by a KMS key using any of the following operations:
    *
    * \- Encrypt \- GenerateDataKey \- GenerateDataKeyPair \- GenerateDataKeyWithoutPlaintext \-
    * GenerateDataKeyPairWithoutPlaintext
    *
    * You can use this operation to decrypt ciphertext that was encrypted under a symmetric encryption KMS key or an
    * asymmetric encryption KMS key. When the KMS key is asymmetric, you must specify the KMS key and the encryption
    * algorithm that was used to encrypt the ciphertext.
    *
    * The Decrypt operation also decrypts ciphertext that was encrypted outside of KMS by the public key in an KMS
    * asymmetric KMS key. However, it cannot decrypt symmetric ciphertext produced by other libraries, such as the
    * Amazon Web Services Encryption SDK or Amazon S3 client-side encryption. These libraries return a ciphertext format
    * that is incompatible with KMS.
    *
    * If the ciphertext was encrypted under a symmetric encryption KMS key, the KeyId parameter is optional. KMS can get
    * this information from metadata that it adds to the symmetric ciphertext blob. This feature adds durability to your
    * implementation by ensuring that authorized users can decrypt ciphertext decades after it was encrypted, even if
    * they've lost track of the key ID. However, specifying the KMS key is always recommended as a best practice. When
    * you use the KeyId parameter to specify a KMS key, KMS only uses the KMS key you specify. If the ciphertext was
    * encrypted under a different KMS key, the Decrypt operation fails. This practice ensures that you use the KMS key
    * that you intend.
    */
  inline def decryptMessage(
      keyId: String,
      algorithm: String,
      bytes: Array[Byte]
  )(using
      AwsClient
  ): Array[Byte] =
    AwsClient.invoke(s"decryptMessage") {
      summon[AwsClient].kms
        .decrypt(
          DecryptRequest
            .builder()
            .keyId(keyId)
            .encryptionAlgorithm(algorithm)
            .ciphertextBlob(SdkBytes.fromByteArrayUnsafe(bytes))
            .build()
        )
        .plaintext()
        .asByteArrayUnsafe()
    }

}
