/*
 * Encog(tm) Core v3.0 - Java Version
 * http://www.heatonresearch.com/encog/
 * http://code.google.com/p/encog-java/
 
 * Copyright 2008-2011 Heaton Research, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *   
 * For more information on Heaton Research copyrights, licenses 
 * and trademarks visit:
 * http://www.heatonresearch.com/copyright
 */
package org.encog.ml.kmeans;

import java.util.ArrayList;
import java.util.List;

import org.encog.ml.MLCluster;
import org.encog.ml.data.MLData;
import org.encog.ml.data.MLDataSet;
import org.encog.ml.data.basic.BasicMLDataSet;

/**
 * Holds a cluster of MLData items that have been clustered 
 * by the KMeansClustering class.
 */
public class KMeansCluster implements MLCluster {

	/**
	 * The centroid.
	 */
	private Centroid centroid;
	
	/**
	 * The sum square.
	 */
	private double sumSqr;
	
	/**
	 * The contents of the cluster.
	 */
	private final List<MLData> data = new ArrayList<MLData>();

	/**
	 * Add to the cluster.
	 * @param pair The pair to add.
	 */
	@Override
	public final void add(final MLData pair) {
		this.data.add(pair);
		calcSumOfSquares();
	}

	/**
	 * Calculate the sum of squares.
	 */
	public final void calcSumOfSquares() { 
		final int size = this.data.size();
		double temp = 0;
		for (int i = 0; i < size; i++) {
			temp += KMeansClustering.calculateEuclideanDistance(this.centroid,
					(this.data.get(i)));
		}
		this.sumSqr = temp;
	}

	/**
	 * Create a dataset from the clustered data.
	 * @return The dataset.
	 */
	@Override
	public final MLDataSet createDataSet() {
		final MLDataSet result = new BasicMLDataSet();

		for (final MLData dataItem : this.data) {
			result.add(dataItem);
		}

		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final MLData get(final int pos) {
		return this.data.get(pos);
	}

	/**
	 * @return The centroid.
	 */
	public final Centroid getCentroid() {
		return this.centroid;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<MLData> getData() {
		return this.data;
	}

	/**
	 * @return The sum of squares.
	 */
	public final double getSumSqr() {
		return this.sumSqr;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void remove(final MLData pair) {
		this.data.remove(pair);
		calcSumOfSquares();
	}

	/**
	 * Set the centroid.
	 * @param c The new centroid.
	 */
	public final void setCentroid(final Centroid c) {
		this.centroid = c;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int size() {
		return this.data.size();
	}

}
