/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea

import endea.http._
import endea.internal.http._
import java.net.URL
import java.util.Date
import java.util.zip.CRC32
import javax.servlet.http._
import resource.{ Resource => _, _ }
import scala.collection.mutable.HashMap

object Resource {

  private val map = new HashMap[String, Resource]()

  def get(name: String): Option[Resource] = {

    map.get(name) match {
      case s: Some[_] => s
      case _ =>
        val url = Resource.getClass().getResource(name)
        if (url == null)
          None
        else {
          val resource = new Resource(name, url)
          map(name) = resource
          Some(resource)
        }
    }
  }

  def apply(context: AnyRef, name: String): Option[Resource] = {
    get(context.getClass().getName(), name)
  }

  def apply[T](name: String)(implicit context: Manifest[T]): Option[Resource] = {
    get(context.toString(), name)
  }

  private def get(className: String, name: String): Option[Resource] = {

    val index = className.lastIndexOf('.')

    var bname =
      if (index == -1) "/"
      else "/" + className.substring(0, index).replace('.', '/') + "/"
    bname += name

    val resource = get(bname)
    if (!resource.isDefined)
      // TODO
      print("? " + bname)

    resource
  }
}

@Id(0x25714821)
class Resource private (val name: String, private val url: URL) extends Entity {

  lazy val path = Controller.encode(this)

  lazy val crc = {

    val crc32 = new CRC32()

    val buffer = new Array[Byte](512)
    for (input <- managed(url.openStream())) {
      Stream.continually(input.read(buffer)).
        takeWhile(_ != -1).
        foreach(crc32.update(buffer, 0, _))
    }

    crc32.getValue().toHexString
  }

  lazy val _type = {

    if (name.endsWith(".css"))
      "text/css"
    else if (name.endsWith(".ico"))
      "image/vnd.microsoft.icon"
    else if (name.endsWith(".gif"))
      "image/gif"
    else if (name.endsWith(".png"))
      "image/png"
    else "..."
  }

  def write(event: Event[_]) {
    event.write(path)
  }
}