/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal

import java.io._

class ByteEncoder(private val outputStream: OutputStream) {

  def writeInt(int: Int) {

    write(int >>> 24)
    write(int >>> 16)
    write(int >>> 8)
    write(int)
  }

  def writeVarInt(int: Int) {

    var intCopy = int
    while (true) {
      if ((intCopy & ~0x7F) == 0) {
        write(intCopy)
        return
      } else {
        write((intCopy & 0x7F) | 0x80)
        intCopy >>>= 7
      }
    }
  }

  def writeInvVarInt(int: Int) {
    writeVarInt(~int)
  }

  def writeLong(long: Long) {

    write(long >>> 56)
    write(long >>> 48)
    write(long >>> 40)
    write(long >>> 32)
    write(long >>> 24)
    write(long >>> 16)
    write(long >>> 8)
    write(long)
  }

  def writeVarLong(long: Long) {

    var longCopy = long
    while (true) {
      if ((longCopy & ~0x7FL) == 0) {
        write(longCopy)
        return
      } else {
        write((longCopy & 0x7F) | 0x80)
        longCopy >>>= 7
      }
    }
  }

  def writeInvVarLong(long: Long) {
    writeVarLong(~long)
  }

  def writeFloat(float: Float) {
    writeInt(java.lang.Float.floatToIntBits(float))
  }

  def writeDouble(double: Double) {
    writeLong(java.lang.Double.doubleToLongBits(double))
  }

  def writeBoolean(boolean: Boolean) {
    write(if (boolean) 1 else 0)
  }

  def writeString(string: String) {
    val bytes = string.getBytes("UTF-8")
    writeVarInt(bytes.length)
    writeBytes(bytes)
  }

  @inline
  def write(byte: Int) {
    outputStream.write(byte)
  }

  @inline
  def writeBytes(bytes: Array[Byte]) {
    outputStream.write(bytes)
  }

  @inline
  private def write(byte: Long) {
    outputStream.write(byte.asInstanceOf[Byte])
  }
}