/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal

import java.lang.reflect._

import scala.collection.mutable._
import scala.io.Source

import endea._

object MetaEntity {

  private val nameToMetaEntity = new HashMap[String, MetaEntity]
  private val idToMetaEntity = new HashMap[Int, MetaEntity]

  lazy val id = {
    val id = classOf[Entity].getDeclaredField("id")
    id.setAccessible(true)
    id
  }

  for (clazz <- Entity.all) {
    val metaEntity = new MetaEntity(clazz)
    nameToMetaEntity += clazz.getName() -> metaEntity
    idToMetaEntity += metaEntity.id -> metaEntity
  }

  def get[E <: Entity](implicit manifest: Manifest[E]) = {

    getByClassName(manifest.toString()) match {
      case Some(metaEntity) => metaEntity
      case _ => throw new Exception("Register " + manifest.toString())
    }
  }

  def getByClassName(entityClass: String): Option[MetaEntity] = nameToMetaEntity.get(entityClass)

  def getById(id: Int): Option[MetaEntity] = idToMetaEntity.get(id)

  def getByEntity(entity: Entity): MetaEntity = {

    getByClassName(entity.getClass().getName()) match {
      case Some(metaEntity) => metaEntity
      case _ => throw new Exception("Register " + entity.getClass().getName())
    }
  }

  def all() = nameToMetaEntity.values
}

class MetaEntity(val entityClass: Class[_]) {

  private val annotation = entityClass.getAnnotation(classOf[Id])
  if (annotation == null)
    throw new IllegalStateException("@Id not found for: " + entityClass.getName())

  val id = annotation.value()

  val name = entityClass.getSimpleName()

  private lazy val instantiator = new org.objenesis.ObjenesisStd().getInstantiatorOf(entityClass)

  def newInstance(): Entity = instantiator.newInstance().asInstanceOf[Entity]

  lazy val attributes: List[Attribute] = {

    var attributes: List[Attribute] = Nil

    var clazz = entityClass
    while (clazz != classOf[Entity]) {

      for (
        field <- clazz.getDeclaredFields();
        if (!Modifier.isStatic(field.getModifiers()))
      ) {
        val annotation = field.getAnnotation(classOf[Id])
        if (annotation == null)
          throw new RuntimeException("@Id? : " + field)
        attributes = new Attribute(field) :: attributes
      }

      clazz = clazz.getSuperclass()
    }

    attributes
  }

  def attribute(id: Int): Option[Attribute] = {

    for (attribute <- attributes) {
      if (attribute.id == id)
        return Some(attribute)
    }

    None
  }
}
