/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.data

import endea._
import endea.internal.{ Attribute => _, _ }
import endea.internal.DynamicClass._

import java.io._
import java.util._
import org.objectweb.asm._
import org.objectweb.asm.Opcodes._
import org.objectweb.asm.util._

// TODO better VerifyError exception , metadatada fiels empty
object DynamicDataDecoder {

  
  def get[D <: Data](dataClass: Class[D], fields: Array[Field]): DataDecoder[D] = {

    val className = dataClass.getName() + "Decoder"

    try {
      val encoderClass = DynamicClassLoader.loadClass(className,
        Unit => writeEncoder(className, dataClass, fields))
      encoderClass.newInstance().asInstanceOf[DataDecoder[D]]
    } catch {
      case e: VerifyError =>
        val reader = new ClassReader(writeEncoder(className, dataClass, fields))
        reader.accept(new TraceClassVisitor(new PrintWriter(System.out)),
          new Array[Attribute](0), ClassReader.SKIP_DEBUG)

        throw e
    }
  }

  private def writeEncoder[D <: Data](className: String, dataClass: Class[D],
    fields: Array[Field]): Array[Byte] = {

    val bClassName = binaryClassName(className)
    val bDataClassName = binaryClassName(dataClass)
    val instanceSignature = methodSignature(Array(classOf[ByteDecoder]), Some(dataClass))

    val cw = new ClassWriter(ClassWriter.COMPUTE_FRAMES)

    cw.visit(V1_5, ACC_PUBLIC + ACC_SUPER, bClassName,
      "Ljava/lang/Object;Lendea/internal/data/DataDecoder<" + classSignature(dataClass) + ">;",
      "java/lang/Object",
      Array(binaryClassName(classOf[DataDecoder[_]])))

    {
      val mv = cw.visitMethod(ACC_PUBLIC, "<init>", "()V", null, null)
      mv.visitVarInsn(ALOAD, 0)
      mv.visitMethodInsn(INVOKESPECIAL, "java/lang/Object", "<init>", "()V")
      mv.visitInsn(RETURN)
      mv.visitMaxs(0, 0)
      mv.visitEnd()
    }

    {
      val mv = cw.visitMethod(ACC_PUBLIC + ACC_BRIDGE, "instance",
        "(Lendea/internal/ByteDecoder;)Lendea/Data;", null, null)
      mv.visitVarInsn(ALOAD, 0)
      mv.visitVarInsn(ALOAD, 1)
      mv.visitMethodInsn(INVOKEVIRTUAL, bClassName, "instance", instanceSignature)
      mv.visitInsn(ARETURN)
      mv.visitMaxs(0, 0)
      mv.visitEnd()
    }

    {
      val mv = cw.visitMethod(ACC_PUBLIC, "instance", instanceSignature, null, null)
      var index = 2

      import Field._
      for (field <- fields) {
        mv.visitVarInsn(ALOAD, 1)
        field.clazz match {
          case IC =>
            mv.visitMethodInsn(INVOKEVIRTUAL, "endea/internal/ByteDecoder", "readVarInt", "()I")
            mv.visitVarInsn(ISTORE, index)
          case SC =>
            mv.visitMethodInsn(INVOKEVIRTUAL, "endea/internal/ByteDecoder", "readString",
              "()Ljava/lang/String;")
            mv.visitVarInsn(ASTORE, index)
        }
        index += 1
      }

      mv.visitTypeInsn(NEW, bDataClassName)
      mv.visitInsn(DUP)

      // TODO compare with fields and ...
      index = 2
      for (field <- getConstructorFields(dataClass)) {
        field.clazz match {
          case IC =>
            mv.visitVarInsn(ILOAD, index)
          case SC =>
            mv.visitVarInsn(ALOAD, index)
        }
        index += 1
      }

      mv.visitMethodInsn(INVOKESPECIAL, bDataClassName, "<init>", "(Ljava/lang/String;IIIIII)V")
      mv.visitInsn(ARETURN)

      mv.visitMaxs(0, 0)
      mv.visitEnd()
    }

    cw.toByteArray()
  }

  def getConstructorFields[D <: Data](dataClass: Class[D]): Array[Field] = {

    val constructor = dataClass.getDeclaredConstructors()(0)
    val annotations = constructor.getParameterAnnotations()
    val types = constructor.getParameterTypes()

    val fields = new scala.collection.mutable.ArrayBuffer[Field]()

    for (i <- 0 to types.length - 1) {
      fields += new Field(i + 1, "", types(i))
    }

    fields.toArray
  }
}