/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.data

import endea._
import endea.internal._
import endea.internal.DynamicClass._

import java.util._
import org.objectweb.asm._
import org.objectweb.asm.Opcodes._

object DynamicDataEncoder {

  def get[D <: Data](dataClass: Class[D]): DataEncoder[D] = {

    val className = dataClass.getName() + "Encoder"
    val encoderClass = DynamicClassLoader.loadClass(className,
      Unit => writeEncoder(className, dataClass))

    encoderClass.newInstance().asInstanceOf[DataEncoder[D]]
  }

  private def writeEncoder[D <: Data](className: String, dataClass: Class[D]): Array[Byte] = {

    val cw = new ClassWriter(ClassWriter.COMPUTE_FRAMES)

    val bClassName = binaryClassName(className)

    cw.visit(V1_6, ACC_PUBLIC + ACC_SUPER, bClassName,
      "Ljava/lang/Object;Lendea/internal/data/DataEncoder<" + classSignature(dataClass) + ">;",
      "java/lang/Object",
      Array(binaryClassName(classOf[DataEncoder[_]])))

    {
      val mv = cw.visitMethod(ACC_PUBLIC, "<init>", "()V", null, null)
      mv.visitVarInsn(ALOAD, 0)
      mv.visitMethodInsn(INVOKESPECIAL, "java/lang/Object", "<init>", "()V")
      mv.visitInsn(RETURN)
      mv.visitMaxs(0, 0)
      mv.visitEnd()
    }

    val bDataClassName = binaryClassName(dataClass)
    val writeSignature = methodSignature(Array(classOf[ByteEncoder], dataClass), None)

    {
      val mv = cw.visitMethod(ACC_PUBLIC + ACC_BRIDGE + ACC_SYNTHETIC, "write",
        "(Lendea/internal/ByteEncoder;Lendea/Data;)V", null, null)
      mv.visitVarInsn(ALOAD, 0)
      mv.visitVarInsn(ALOAD, 1)
      mv.visitVarInsn(ALOAD, 2)
      mv.visitTypeInsn(CHECKCAST, bDataClassName)
      mv.visitMethodInsn(INVOKEVIRTUAL, bClassName, "write", writeSignature)
      mv.visitInsn(RETURN)
      mv.visitMaxs(0, 0)
      mv.visitEnd()
    }

    {
      val mv = cw.visitMethod(ACC_PUBLIC, "write", writeSignature, null, null)

      def writeString(mv: MethodVisitor, name: String) {
        mv.visitMethodInsn(INVOKEVIRTUAL, bDataClassName, name, "()Ljava/lang/String;")
        mv.visitMethodInsn(INVOKEVIRTUAL, "endea/internal/ByteEncoder", "writeString", "(Ljava/lang/String;)V")
      }

      def writeInt(mv: MethodVisitor, name: String) {
        mv.visitMethodInsn(INVOKEVIRTUAL, bDataClassName, name, "()I")
        mv.visitMethodInsn(INVOKEVIRTUAL, "endea/internal/ByteEncoder", "writeVarInt", "(I)V")
      }

      for (field <- MetaData.get(dataClass).fields) {

        mv.visitVarInsn(ALOAD, 1);
        mv.visitVarInsn(ALOAD, 2);

        import Field._
        field.clazz match {
          case IC => writeInt(mv, field.name)
          case SC => writeString(mv, field.name)
        }
      }

      mv.visitInsn(RETURN)
      mv.visitMaxs(0, 0)
      mv.visitEnd()
    }

    cw.toByteArray()
  }
}
