/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.entity

import endea._
import endea.internal._

import java.io._
import java.util.concurrent._

import scala.collection.mutable._

import com.sleepycat.je._

object EntityStore {

  val class_attribute = 1

  private[entity] val entities = new ConcurrentHashMap[Long, Entity]

  private lazy val environment: Environment = {

    val envConfig = new EnvironmentConfig()
    envConfig.setAllowCreate(true)
    envConfig.setTransactional(true)

    val directory = new File(Endea.path, "store/entity")
    if (!directory.exists())
      directory.mkdirs()

    new Environment(directory, envConfig)
  }

  private lazy val database: Database = {

    val dbConfig = new DatabaseConfig()
    dbConfig.setAllowCreate(true)
    dbConfig.setTransactional(true)

    environment.openDatabase(null, "entities", dbConfig);
  }

  private[entity] val indexes = new ConcurrentHashMap[Int, Database]

  def put[E <: Entity](entity: E): Boolean = {

    val transient = entity.id == 0
    val transaction = environment.beginTransaction(null, null)

    try {

      val id = if (transient) sequence.get(transaction, 1) else entity.id

      val success = database.put(transaction, new DatabaseEntry(Key.encodeLong(id)),
        new DatabaseEntry(entity.bytes))

      if (success != OperationStatus.SUCCESS) {
        transaction.abort()
        false

      } else {

        val metaEntity = MetaEntity.getByEntity(entity)
        index(class_attribute, entity, metaEntity.id)
        for (attribute <- metaEntity.attributes)
          if (attribute.indexed) {
            val value = attribute.get(entity)
            index(attribute.id, entity, value)
          }

        transaction.commit()

        // TODO
        MetaEntity.id.set(entity, id)
        if (transient)
          entities.put(entity.id, entity)

        true
      }

    } catch {
      case e: Exception =>
        if (transient)
          MetaEntity.id.set(entity, 0)
        transaction.abort()
        throw e
    }
  }

  def get[E <: Entity](id: Long)(implicit manifest: Manifest[E]): Option[E] = {

    val cMf = ClassManifest.fromClass(classOf[Entity])

    var entity = entities.get(id)

    if (entity == null) {

      val keyEntry = new DatabaseEntry(Key.encodeLong(id))
      val dataEntry = new DatabaseEntry()

      val status = database.get(null, keyEntry, dataEntry, LockMode.DEFAULT)

      if (status == OperationStatus.SUCCESS) {
        entity = Entity.instance(dataEntry.getData())
        MetaEntity.id.set(entity, id)
        entities.put(entity.id, entity)
      }
    }

    val nothing = Manifest.Nothing

    if (entity != null && manifest.erasure.isAssignableFrom(entity.getClass()))
      Some(entity).asInstanceOf[Some[E]]
    else
      None
  }

  private[entity] def getIndex(attribute: Int): Database = {

    var index = indexes.get(attribute)
    if (index != null)
      return index

    // TODO sync

    val dbConfig = new DatabaseConfig()
    dbConfig.setAllowCreate(true)
    dbConfig.setSortedDuplicates(true)

    index = environment.openDatabase(null, attribute.toHexString, dbConfig);
    indexes.put(attribute, index)

    index
  }

  private def index(attribute: Int, entity: Entity, key: Any) {

    getIndex(attribute).
      put(null, new DatabaseEntry(Key.encode(key)), new DatabaseEntry(Key.encodeLong(entity.id)))
  }

  lazy val sequence: Sequence = {

    import com.sleepycat.bind.tuple._

    val config = new SequenceConfig()
    config.setInitialValue(1)
    config.setRange(1, Long.MaxValue)
    config.setCacheSize(100)
    config.setAutoCommitNoSync(true)
    config.setAllowCreate(true)

    val entry = new DatabaseEntry()
    StringBinding.stringToEntry("sequence", entry)

    database.openSequence(null, entry, config)
  }

  private def getDatabaseFile(): File = {

    val directory = new File("target/database")
    if (!directory.exists())
      directory.mkdirs()

    return directory
  }

  def stop() = {

    //indexes.values.foreach(_.close)

    if (database != null)
      database.close();

    if (environment != null)
      environment.close();
  }
}