/*
 * Copyright 2012 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.html

import endea._
import endea.http._

abstract class HtmlAction[E <: Entity](
  implicit manifest: Manifest[E]) extends HttpAction[E] {

  def css: Option[Resource]

  def js: Option[Resource]

  def controls(event: Event[E]): Seq[Control] = Nil

  def write(event: Event[E])

  def doGet(event: Event[E]) {

    event.write("""<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8" />
  <title>""")
    event.write(HtmlConfig.title)
    event.write("""</title>
  <link rel="shorcut icon" type="image/x-icon" href="""")
    write(event, HtmlConfig.favicon, """"/>""")
    writeCss(event, HtmlConfig.css)
    writeCss(event, css)
    event.write("""<link rel="apple-touch-icon" href="/apple-touch-icon.png"/>
</head>
<body>
  <div id="nav">
""")
    nav(event)
    event.write("""  </div>
  <div id="main">
""")
    main(event)
    event.write("""  </div>
""")
    writeJs(event, HtmlConfig.jquery)
    writeJs(event, HtmlConfig.jquery_ui)
    writeJs(event, js)
    event.write("""</body>
</html>""")
  }

  private def nav(event: Event[E]) {

    for (control <- HtmlConfig.controls) {
      if (control.contribute(event)) {
        event.write("<a href=\"")
        event.write(control.url(event))
        event.write("\" title=\"")
        event.write(control.name)
        event.write("\">")
        ImageView.write(event, control.icon(event), 48)
        event.write("</a>\n")
      }
    }

    for (control <- controls(event)) {
      if (control.contribute(event)) {
        event.write("<a href=\"")
        event.write(control.url(event))
        event.write("\" title=\"")
        event.write(control.name)
        event.write("\">")
        ImageView.write(event, control.icon(event), 48)
        event.write("</a>\n")
      }
    }

  }

  def main(event: Event[E]) {

    write(event)
  }

  private def writeCss(event: Event[E], option: Option[Resource]) {

    option match {
      case Some(resource) =>
        event.write("""<link rel="stylesheet" type="text/css" href="""")
        event.write(resource.path)
        event.write("""" />
""")
      case _ =>
    }
  }

  private def writeJs(event: Event[E], option: Option[Resource]) {

    option match {
      case Some(resource) =>
        event.write("""<script type="text/javascript" language="javascript" src="""")
        event.write(resource.path)
        event.write(""""></script>
""")
      case _ =>
    }
  }

  private def write(event: Event[E], resource: Option[Resource], string: String) {
    ResourceView.write(event, resource)
    event.write(string)
  }
}