/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal

object DynamicClass {

  private lazy val PrimitiveSigMap = Map(
    classOf[Boolean].getName -> "Z",
    classOf[Byte].getName -> "B",
    classOf[Char].getName -> "C",
    classOf[Short].getName -> "S",
    classOf[Int].getName -> "I",
    classOf[Long].getName -> "J",
    classOf[Float].getName -> "F",
    classOf[Double].getName -> "D",
    classOf[Unit].getName -> "V",
    "void" -> "V")

  def loadClass(className: String, classBytes: Array[Byte]): Class[_] = {
    loadClass(ClassLoader.getSystemClassLoader(), className, classBytes)
  }

  def loadClass(classLoader: ClassLoader, className: String, classBytes: Array[Byte]): Class[_] = {

    val method = Class.forName("java.lang.ClassLoader").
      getDeclaredMethod("defineClass", classOf[String], classOf[Array[Byte]], classOf[Int], classOf[Int])
    method.setAccessible(true)

    method.invoke(classLoader, className, classBytes,
      new java.lang.Integer(0),
      new java.lang.Integer(classBytes.length)).
      asInstanceOf[Class[_]]
  }

  def classSignature(cls: Class[_]): String = {

    if (cls.isArray)
      "[" + classSignature(cls.getComponentType)

    else if (cls.isPrimitive) {
      val s = PrimitiveSigMap.get(cls.getName)
      if (s == None)
        throw new Exception("Can't map class \"" + cls.getName + "\" " +
          "to signature.")
      s.get
    } else
      "L" + binaryClassName(cls.getName) + ";"
  }

  def methodSignature(paramTypes: Array[Class[_]], returnType: Option[Class[_]]) = {

    val paramSig =
      if ((paramTypes == null) || (paramTypes.length == 0))
        ""
      else
        paramTypes.map(pt => classSignature(pt)).mkString("")

    val returnTypeSignature = returnType match {
      case Some(returnType) => classSignature(returnType)
      case None => "V"
    }

    "(" + paramSig + ")" + returnTypeSignature
  }

  def binaryClassName(clazz: Class[_]): String =
    binaryClassName(clazz.getName())

  def binaryClassName(className: String): String =
    className.replaceAll("""\.""", "/")
}