/*
 * Copyright 2012 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.auth

import endea._
import endea.auth._
import endea.html._
import endea.http._
import org.brickred.socialauth._
import org.brickred.socialauth.util.SocialAuthUtil

object OpenIdAction extends HtmlAction[Type[User]] {

  override val name = "open-id"

  val css = Resource(OpenIdAction, "open-id.css")
  val js = None

  private val uri = Controller.encode(OpenIdAction)
  private val _provider = "provider"
  private val _auth_manager = "OAuthManager"
  private val success_url = "http://www.autoexamen.com/~me"

  private val signIn = new OAuthProvider("Sign In", Image(OpenIdAction, "sign-in.png"),
    Controller.encode(SignInAction))

  private lazy val authConfig = {

    val authConfig = SocialAuthConfig.getDefault()
    authConfig.load()
    authConfig
  }

  override def doGet(event: Event[Type[User]]) {

    event.parameter(_provider) match {
      case Some(provider) => auth(event, provider)
      case None => super.doGet(event)
    }
  }

  def write(event: Event[Type[User]]) {

    event.write("""<div class="login-panel">
""")
    for (provider <- AuthConfig.providers)
      write(event, provider)
    write(event, signIn)
    event.write("""</div>
""")
  }

  private def write(event: Event[Type[User]], provider: OAuthProvider) {
    event.write("""<div class="provider"><a href="""")
    event.write(provider.uri)
    event.write("\">")
    ImageView.write(event, provider.icon, 64)
    event.write("</a><p>")
    event.write(provider.name)
    event.write("""</p></div>
""")
  }

  private def auth(event: Event[Type[User]], provider: String) {

    val manager = new SocialAuthManager()
    manager.setSocialAuthConfig(authConfig)
    val authenticationUrl = manager.getAuthenticationUrl(
      provider, success_url);

    event.httpRequest.getSession().setAttribute(_auth_manager, manager)
    event.httpResponse.sendRedirect(authenticationUrl)
  }
}