/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.data

import endea._
import endea.internal._
import endea.internal.data.MetaData._

import java.lang.reflect.{ Field => JField }

object Field {

  val I = 0
  val IC = classOf[Int]
  val L = 1
  val LC = classOf[Long]
  val S = 2
  val SC = classOf[String]

  def read(decoder: ByteDecoder): Array[Field] = {

    if (!decoder.hasNext())
      return new Array[Field](0)

    val numOfFields = decoder.readVarInt()
    val fields = new Array[Field](numOfFields)

    for (i <- 0 to numOfFields - 1) {
      val id = decoder.readVarInt()
      val clazz = decoder.readVarInt() match {
        case I => IC
        case S => SC
      }
      fields(i) = new Field(id, "", clazz)
    }

    fields
  }

  def write[D <: Data](encoder: ByteEncoder, dataClass: Class[D]) {

    val metaData = MetaData.get(dataClass)

    encoder.writeVarInt(metaData.fields.length)

    for (field <- metaData.fields) {
      encoder.writeVarInt(field.id)
      encoder.writeVarInt(field.clazz match {
        case IC => I
        case SC => S
      })
    }
  }
}

class Field(
  val id: Int,
  val name: String,
  val clazz: Class[_]) {

  def this(id: Int, field: JField) = this(id, field.getName, field.getType)

  override def toString() =
    "@" + id + " " + name + ":" + clazz.getSimpleName()

  override def equals(that: Any) =
    that match {
      case that: Field =>
        id == that.id && clazz == that.clazz
      case _ => false
    }
}