/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.http

import endea._
import endea.html._
import endea.http._
import endea.internal._
import endea.internal.entity._
import endea.internal.html._
import endea.internal.http._

import scala.collection.mutable.HashMap
import scala.collection.mutable.ArrayBuffer

object Controller {

  private val classToName = new HashMap[String, String]
  private val nameToClass = new HashMap[String, Class[_ <: Entity]]()

  private val actions = new HashMap[String, HttpAction[_]]

  for (entityClass <- MetaEntity.all)
    addEntityClass(entityClass)

  for (module <- Module.modules; action <- module.actions) {
    val key = action.className + "/" + action.name
    actions += (key -> action)
  }

  def encode[E <: Entity](entity: E, action: HttpAction[E] = DefaultAction) = {

    var url = "/" + entity.id
    if (!action.name.equals(HttpAction.defaultAction))
      url += "/" + action.name
    url
  }

  def encodeSession[E <: Entity](key: String, action: HttpAction[E] = DefaultAction) = {

    var url = "/~" + key
    if (!action.name.equals(HttpAction.defaultAction))
      url += "/" + action.name
    url
  }

  def encode[E <: Entity](action: HttpAction[Type[E]])(implicit manifest: Manifest[E]) = {

    val name = classToName.getOrElseUpdate(manifest.toString(),
      addEntityClass(manifest.erasure.asInstanceOf[Class[_ <: Entity]]))

    var url = "/" + name
    if (!action.name.equals(HttpAction.defaultAction))
      url += "/" + action.name
    url
  }

  def encode(resource: Resource) = {
    "/resource/" + resource.crc + resource.name
  }

  def decode(requestURI: String, session: Session): (Entity, HttpAction[_ <: Entity]) = {

    if (requestURI.length() == 1) {

      val key = classOf[Home].getName()
      var home = session.get[Home](key) match {
        case Some(home) => home
        case _ =>
          val home = new Home
          session.put(key, home)
          home
      }

      (home, HomeAction)

      // resource
    } else if (requestURI.startsWith("/resource/")) {

      val path = requestURI.substring(requestURI.indexOf('/', 10), requestURI.length())

      return Resource.get(path) match {
        case Some(resource) => (resource, ResourceAction)
        case _ => (HttpError(), HttpErrorAction)
      }

      // session
    } else if (requestURI.charAt(1) == '~') {

      val (key, action) = extract(requestURI, 2)
      session.get[Entity](key) match {
        case Some(entity) => (entity, getAction(entity.getClass(), action))
        case _ => (new HtmlError("Entity not found"), HtmlErrorAction)
      }
      // entity
    } else if (requestURI.charAt(1).isDigit) {

      val (id, action) = extract(requestURI)
      Entity.get[Entity](id.toLong) match {
        case Some(entity) => (entity, getTypeAction(entity.getClass(), action))
        case _ => (new HtmlError("Entity not found"), HtmlErrorAction)
      }

      // class
    } else {

      val (className, action) = extract(requestURI)
      nameToClass.get(className) match {
        case Some(clazz) => (Type(clazz), getTypeAction(clazz, action))
        case _ => (new HtmlError("Class not found"), HtmlErrorAction)
      }
    }
  }

  private def extract(uri: String, start: Int = 1): (String, String) = {

    val index = uri.indexOf('/', start)

    if (index == -1) //id
      (uri.substring(start), HttpAction.defaultAction)
    else if (index == (uri.length() - 1)) //id/
      (uri.substring(start, index), HttpAction.defaultAction)
    else //id/action
      (uri.substring(start, index), uri.substring(index + 1))
    // TODO ...
  }

  private def getAction[E <: Entity](clazz: Class[_], action: String): HttpAction[E] = {

    // TODO
    if (clazz == null)
      throw new RuntimeException("Unknow action: " + action)

    actions.get(clazz.getName + "/" + action) match {
      case Some(action) => action.asInstanceOf[HttpAction[E]]
      case _ => getAction(clazz.getSuperclass(), action)
    }
  }

  private def getTypeAction[E <: Entity](clazz: Class[_], action: String): HttpAction[E] = {

    actions.get("endea.http.Type[" + clazz.getName + "]/" + action) match {
      case Some(action) => action.asInstanceOf[HttpAction[E]]
      case _ => getAction(clazz.getSuperclass(), action)
    }
  }

  private def addEntityClass(entityClass: Class[_ <: Entity]) = {

    val array = new ArrayBuffer[Char]()
    var previousUpper = true

    for (char <- entityClass.getSimpleName()) {
      val upper = char.isUpper
      if (upper && !previousUpper)
        array += '-'
      array += (if (upper) char.toLower else char)
      previousUpper = upper
    }

    val name = new String(array.toArray)
    classToName += (entityClass.getName() -> name)
    nameToClass += (name -> entityClass)
    name
  }
}