/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.http

import endea._
import endea.internal.http._
import java.io._
import java.net.URL
import java.util.Date
import java.util.zip.CRC32
import javax.servlet.http._
import resource.{ Resource => _, _ }
import scala.collection.mutable.HashMap

import java.awt.color._
import java.awt.image._

import org.imgscalr._
import javax.imageio.ImageIO

object Resource {

  private val imageDir = new File(Endea.tmpPath, "image")

  private val map = new HashMap[String, Resource]()

  def get(name: String): Option[Resource] = {

    map.get(name) match {
      case some: Some[_] => some
      case _ =>
        if (name.startsWith("/g/") || Character.isDigit(name.charAt(2))) {
          createImage(name)
        } else {
          val url = Resource.getClass().getResource(name)
          if (url == null) None else add(name, url)
        }
    }
  }

  def apply(context: AnyRef, name: String): Option[Resource] = {
    get(context.getClass().getName(), name)
  }

  def apply[T](name: String)(implicit context: Manifest[T]): Option[Resource] = {
    get(context.toString(), name)
  }

  private def get(className: String, name: String): Option[Resource] = {

    val index = className.lastIndexOf('.')

    var bname =
      if (index == -1) "/"
      else "/" + className.substring(0, index).replace('.', '/') + "/"
    bname += name

    val resource = get(bname)
    if (!resource.isDefined)
      // TODO
      print("? " + bname)

    resource
  }

  private def add(name: String, url: URL) = {

    val resource = new Resource(name, url)
    map += name -> resource
    Some(resource)
  }

  private def createImage(name: String): Option[Resource] = {

    var index = name.indexOf('/', 1)
    val image = name.substring(index)

    get(image) match {
      case Some(resource) =>

        val char = name.charAt(2)
        if (char == 'g') {
          gray(resource, name)
        } else {
          var size = name.substring(1, index)
          if (size.last != 'g')
            resize(resource, name, size)
          else {
            size = size.dropRight(1)

            val newResource = get("/" + size + image).get
            //val newResource = resize(resource, "/" + size + image, size).get
            gray(newResource, name)
          }
        }

      case _ => None
    }
  }

  private def resize(resource: Resource, name: String, size: String) = {

    val inputStream = resource.url.openStream()
    try {

      val bufferedImage = ImageIO.read(inputStream)

      val newImage = Scalr.resize(bufferedImage, Scalr.Method.QUALITY, Integer.parseInt(size), Scalr.OP_ANTIALIAS)

      val file = new File(imageDir, name)
      file.getParentFile().mkdirs()

      ImageIO.write(newImage, "png", file)

      add(name, file.toURI().toURL())

    } finally {
      if (inputStream != null)
        inputStream.close()
    }
  }

  private def gray(resource: Resource, name: String) = {

    val inputStream = resource.url.openStream()
    try {
      val bufferedImage = ImageIO.read(inputStream)

      /////

      val ccop = new ColorConvertOp(ColorSpace
        .getInstance(ColorSpace.CS_GRAY), null)
      val grayed = ccop.filter(bufferedImage, null)

      //val grayed = Scalr.apply(resized, Scalr.OP_GRAYSCALE)

      /////

      val file = new File(imageDir, name)
      file.getParentFile().mkdirs()

      // TODO extension
      ImageIO.write(grayed, "png", file)

      add(name, file.toURI().toURL())

    } finally {
      if (inputStream != null)
        inputStream.close()
    }
  }
}

class Resource private (val name: String, val url: URL) extends Entity {

  lazy val path = Controller.encode(this)

  lazy val crc = {

    val crc32 = new CRC32()

    val buffer = new Array[Byte](512)
    for (input <- managed(url.openStream())) {
      Stream.continually(input.read(buffer)).
        takeWhile(_ != -1).
        foreach(crc32.update(buffer, 0, _))
    }

    crc32.getValue().toHexString
  }

  lazy val _type = {

    if (name.endsWith(".css"))
      "text/css"
    else if (name.endsWith(".ico"))
      "image/vnd.microsoft.icon"
    else if (name.endsWith(".gif"))
      "image/gif"
    else if (name.endsWith(".png"))
      "image/png"
    else if (name.endsWith(".js"))
      "text/javascript"
    else ""
  }

  def write(event: Event[_]) {
    event.write(path)
  }
}