/*
 * Copyright 2011 David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal

import java.io._
import java.nio.ByteBuffer
import java.nio.channels.FileChannel

class ByteDecoder(private val inputStream: InputStream) {

  def this(array: Array[Byte]) = this(new ByteArrayInputStream(array))

  def readInt(): Int = {
    (read() << 24) | (read() << 16) | (read() << 8) | read()
  }

  def readVarInt(): Int = {

    var int = read
    if ((int & 0x80) == 0)
      return int

    int = int & 0x7f
    var shift = 7
    while (shift < 32) {
      val byte = read()
      int |= (byte & 0x7F) << shift
      if ((byte & 0x80) == 0)
        return int
      shift += 7
    }

    throw new RuntimeException("Malformed var int")
  }

  def readInvVarInt(): Int = ~readVarInt()

  def readLong(): Long = {

    (read().asInstanceOf[Long] << 56) |
      (read().asInstanceOf[Long] << 48) |
      (read().asInstanceOf[Long] << 40) |
      (read().asInstanceOf[Long] << 32) |
      (read().asInstanceOf[Long] << 24) |
      (read().asInstanceOf[Long] << 16) |
      (read().asInstanceOf[Long] << 8) |
      read().asInstanceOf[Long]
  }

  def readVarLong(): Long = {

    var long = 0L

    var shift = 0
    while (shift < 64) {
      val byte = read()
      long |= (byte & 0x7F).asInstanceOf[Long] << shift
      if ((byte & 0x80) == 0)
        return long
      shift += 7
    }

    throw new RuntimeException("Malformed var long")
  }

  def readInvVarLong(): Long = ~readVarLong()

  def readFloat(): Float = java.lang.Float.intBitsToFloat(readInt)

  def readDouble(): Double = java.lang.Double.longBitsToDouble(readLong())

  def readBoolean(): Boolean = read() != 0

  def readString(): String = {

    val array = new Array[Byte](readVarInt())
    inputStream.read(array)
    new String(array, "UTF-8")
  }

  private var byte = -1

  def read(): Int = {
    if (byte != -1) {
      val byteCopy = byte
      byte = -1
      byteCopy
    } else inputStream.read()
  }

  def hasNext(): Boolean = {

    if (byte != -1)
      true
    else {
      byte = inputStream.read()
      byte != -1
    }
  }
}