/*
 * Copyright (c) 2011, 2012, David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal

import endea._
import java.lang.reflect.Field
import java.lang.reflect.Modifier
import java.io._
import java.util.concurrent.ConcurrentHashMap
import java.security.SecureRandom
import scala.collection.mutable._
import scala.io._
import resource._

object MetaObject {

  private var id = 0
  private val idToMetaObject = new ConcurrentHashMap[Int, MetaObject]

  // TODO use className instead of class
  private val classToMetaObject = new ConcurrentHashMap[Class[_], MetaObject]

  /////

  private val file = new File(Endea.path, "metadata")

  if (!file.exists())
    file.createNewFile()
  else {
    for (metaObject <- load(file)) {
      idToMetaObject.put(metaObject.id, metaObject)
      classToMetaObject.put(metaObject.clazz, metaObject)
    }
  }

  /////

  def getByObject(any: AnyRef) = getByClass(any.getClass())

  // def getByClass[T](implicit manifest: Manifest[T]) = getByClass(manifest.erasure)

  def getByClass(clazz: Class[_]): MetaObject = {

    var metaObject = classToMetaObject.get(clazz)

    if (metaObject == null) {
      file.synchronized {
        metaObject = classToMetaObject.get(clazz)
        if (metaObject == null) {

          val __metaObject = make(clazz)
          save(__metaObject, file)

          idToMetaObject.put(__metaObject.id, __metaObject)
          classToMetaObject.put(__metaObject.clazz, __metaObject)

          metaObject = __metaObject
        }
      }
    }

    metaObject

  }

  def getById(id: Int): Option[MetaObject] = {

    val metaObject = idToMetaObject.get(id)

    if (metaObject != null)
      Some(metaObject)
    else
      None
  }

  private[internal] def make(clazz: Class[_]): MetaObject = {

    def nextId(): Int = {
      id += 1
      id
    }

    val meta = new MetaObject(nextId(), clazz)

    import MetaObject._

    meta.dattributes =
      for (
        field <- clazz.getDeclaredFields();
        if (!Modifier.isTransient(field.getModifiers()))
      ) yield new Attribute(nextId(), field)

    meta
  }

  private[internal] def load(file: File): Seq[MetaObject] = {

    def updateId(value: Int): Int = {
      if (id <= value)
        id = value + 1
      value
    }

    val metaObjects = new ListBuffer[MetaObject]

    var meta: MetaObject = null
    for (line <- Source.fromFile(file).getLines()) {
      if (line.length() > 0) {

        val trimmed = line.trim()
        val index = trimmed.indexOf(' ')

        if (trimmed.charAt(0) == '[') {
          meta = new MetaObject(
            updateId(parseHexInt(line.substring(1, index))),
            Class.forName(line.substring(index).trim))
          metaObjects += meta

        } else {
          meta.loadedAttributes += Tuple2(
            updateId(parseHexInt(line.substring(0, index))),
            line.substring(index).trim())
        }
      }
    }

    metaObjects
  }

  private[internal] def update(meta: MetaObject, file: File) {}

  private[internal] def save(meta: MetaObject, file: File) {

    def writeLine(out: BufferedWriter, id: Int, name: String) {
      out.write(Integer.toHexString(id))
      out.write(" ")
      out.write(name)
      out.newLine()
    }

    for (
      out <- managed(new BufferedWriter(new FileWriter(file, true)))
    ) {

      out.newLine()
      out.write("[")
      writeLine(out, meta.id, meta.clazz.getName)
      out.newLine()

      for (attribute <- meta.attributes)
        writeLine(out, attribute.id, attribute.field.getName)

      out.flush()
    }
  }

  def parseHexInt(string: String): Int = {

    if (string.length() > 8)
      throw new NumberFormatException("For input string: " + string)

    var result = 0

    var index = 0
    while (index < string.length()) {

      result = result << 4
      var digit = Character.digit(string.charAt(index), 16)
      if (digit < 0)
        throw new NumberFormatException("For input string: " + string)

      result |= digit

      index += 1
    }

    result
  }
}

class MetaObject private (

  val id: Int,
  val clazz: Class[_]) {

  private var loadedAttributes = new ListBuffer[(Int, String)]
  private var dattributes: Sequence[Attribute] = _

  lazy val attributes: Sequence[Attribute] = {

    val attributes =
      if (dattributes != null)
        dattributes
      else {
        val attributes = new ListBuffer[Attribute]
        for ((id, name) <- loadedAttributes) {
          attributes += new Attribute(id, clazz.getDeclaredField(name))
        }
        attributes
      }

    attributes
  }

  private lazy val instantiator = new org.objenesis.ObjenesisStd().getInstantiatorOf(clazz)

  def newInstance() = instantiator.newInstance()

  def attribute(id: Int): Option[Attribute] = {

    for (attribute <- attributes)
      if (attribute.id == id)
        return Some(attribute)

    None
  }
}
