/*
 * Copyright (c) 2011, David de Mingo <david@demingo.name>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.entity

import endea._
import endea.internal._
import com.sleepycat.je._
import com.sleepycat.bind.tuple._

object Serial {

  private val HEADER_ID = 1

  private val REF = 0
  private val INT = 1
  private val VAR_INT = 2
  private val INV_VAR_INT = 3
  private val LONG = 4
  private val VAR_LONG = 5
  private val INV_VAR_LONG = 6
  private val FLOAT = 7
  private val DOUBLE = 10
  private val BOOLEAN = 14
  private val STRING = 15

  final def bytes(entity: Entity, writeId: Boolean = false): Array[Byte] = {

    val metaObject = MetaObject.getByObject(entity)

    val byteArrayOS = new ByteArrayOutputStream()
    val encoder = new ByteEncoder(byteArrayOS)

    var header = 0
    if (writeId)
      header = header | HEADER_ID
    encoder.write(header)

    encoder.writeVarInt(metaObject.id)
    if (writeId)
      encoder.writeVarLong(entity.id)

    /////

    for (attribute <- metaObject.attributes) {

      val value: Any = attribute.field.get(entity)
      if (value != null) {
        encoder.writeVarInt(attribute.id)
        value match {
          case entity: Entity =>
            if (entity.id == 0)
              throw new SerialException("Not persisted entity: " + entity)
            else
              encoder.writeVarLong(entity.id)
          case i: Int =>
            if (i < 0 && i > -0xffff) {
              encoder.write(INV_VAR_INT); encoder.writeInvVarInt(i)
            } else if (i >= 0 && i < 0xffff) {
              encoder.write(VAR_INT); encoder.writeVarInt(i)
            } else {
              encoder.write(INT); encoder.writeInt(i)
            }
          case l: Long =>
            if (l < 0L && l > -0xffffffffffffL) {
              encoder.write(INV_VAR_LONG); encoder.writeInvVarLong(l)
            } else if (l >= 0L && l < 0xffffffffffffL) {
              encoder.write(VAR_LONG); encoder.writeVarLong(l)
            } else {
              encoder.write(LONG); encoder.writeLong(l)
            }
          case f: Float => encoder.write(FLOAT); encoder.writeFloat(f)
          case d: Double => encoder.write(DOUBLE); encoder.writeDouble(d)
          case b: Boolean => encoder.write(BOOLEAN); encoder.writeBoolean(b)
          case s: String => encoder.write(STRING); encoder.writeString(s)
        }
      }
    }

    byteArrayOS.toByteArray()
  }

  def instance(bytes: Array[Byte]): Entity = {

    val decoder = new ByteDecoder(bytes)

    val header = decoder.read()

    val entityClassId = decoder.readVarInt()
    val metaEntity = MetaObject.getById(entityClassId).
      getOrElse(throw new RuntimeException("Entity class not found: " + entityClassId))

    val entity = metaEntity.newInstance

    if ((header & HEADER_ID) != 0)
      MetaEntity.id.set(entity, decoder.readVarLong())

    /////

    while (decoder.hasNext) {

      val attributeId = decoder.readVarInt()
      val value = decoder.read() match {
        case INT => decoder.readInt().asInstanceOf[java.lang.Integer]
        case VAR_INT => decoder.readVarInt().asInstanceOf[java.lang.Integer]
        case INV_VAR_INT => decoder.readInvVarInt().asInstanceOf[java.lang.Integer]
        case LONG => decoder.readLong().asInstanceOf[java.lang.Long]
        case VAR_LONG => decoder.readVarLong().asInstanceOf[java.lang.Long]
        case INV_VAR_LONG => decoder.readInvVarLong().asInstanceOf[java.lang.Long]
        case FLOAT => decoder.readFloat().asInstanceOf[java.lang.Float]
        case DOUBLE => decoder.readDouble().asInstanceOf[java.lang.Double]
        case BOOLEAN => decoder.readBoolean().asInstanceOf[java.lang.Boolean]
        case STRING => decoder.readString()
      }

      metaEntity.attribute(attributeId) match {
        case Some(attribute) =>
          attribute.field.set(entity, value)
        case None =>
          println(attributeId + "? " + value)
      }
    }

    entity.asInstanceOf[Entity]
  }
}