/*
 * Copyright (c) 2012, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.entity

import endea._
import endea.internal._
import endea.internal.entity._

import scala.collection.mutable.ArrayBuffer

import com.sleepycat.je._

object StringIndex {

  /**
   * A single FF UTF character is not a valid UTF encoded string.
   */
  val NULL_STRING = new DatabaseEntry(Array[Byte](0xff.asInstanceOf[Byte]))

  def apply[E <: Entity](attribute: String)(implicit manifest: Manifest[E]): StringIndex[E] =
    Index.getIndex[String, E](attribute).asInstanceOf[StringIndex[E]]
}

class StringIndex[E <: Entity](id: String)(implicit manifest: Manifest[E])
  extends Index[String, E](id) {

  override protected def encode(key: String) = {

    if (key == null)
      StringIndex.NULL_STRING
    else {

      val bytes = new ArrayBuffer[Byte]()
      for (char <- key.toCharArray) {
        if ((char >= 0x0001) && (char <= 0x007F)) {
          bytes += char.asInstanceOf[Byte]
        } else if (char > 0x07FF) {
          bytes += (0xE0 | ((char >> 12) & 0x0F)).asInstanceOf[Byte]
          bytes += (0x80 | ((char >> 6) & 0x3F)).asInstanceOf[Byte]
          bytes += (0x80 | ((char >> 0) & 0x3F)).asInstanceOf[Byte]
        } else {
          bytes += (0xC0 | ((char >> 6) & 0x1F)).asInstanceOf[Byte]
          bytes += (0x80 | ((char >> 0) & 0x3F)).asInstanceOf[Byte]
        }
      }

      new DatabaseEntry(bytes.toArray)
    }
  }
}