/*
 * Copyright (c) 2012, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.http

import java.net.URL

import endea.io.io._

import java.net.URL
import java.util.Date

import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

object ClasspathResource {

  val path = "/classpath/"

  private val ONE_YEAR = 31557600000L

  def apply[C](name: String)(implicit context: Manifest[C]): Option[ClasspathResource] =
    make(context.erasure, name)

  def apply(context: AnyRef, name: String): Option[ClasspathResource] =
    make(context.getClass(), name)

  def doGet(request: HttpServletRequest, response: HttpServletResponse) {

    val requestURI = request.getRequestURI()
    val name = requestURI.drop(requestURI.indexOf('/', 11))

    ClasspathResource.getClass().getResource(name) match {
      case null =>
        response.sendError(HttpServletResponse.SC_NOT_FOUND)
      case url =>

        response.setDateHeader("Expires", new Date().getTime() + ONE_YEAR);
        //response.setContentType(resource._type)
        url.openStream().copyTo(response.getOutputStream())
    }

  }

  def _type(name: String) = {

    if (name.endsWith(".css"))
      "text/css"
    else if (name.endsWith(".ico"))
      "image/vnd.microsoft.icon"
    else if (name.endsWith(".gif"))
      "image/gif"
    else if (name.endsWith(".png"))
      "image/png"
    else if (name.endsWith(".js"))
      "text/javascript"
    else ""
  }

  def write(event: Event[_], resource: Option[ClasspathResource]) {

    resource match {
      case Some(resource) => resource.write(event)
      case _ =>
    }
  }

  def make[T](clazz: Class[_], name: String, builder: Builder[T] = DefaultBuilder): Option[T] = {

    val path = getPath(clazz, name)
    val url = ClasspathResource.getClass().getResource(path)

    if (url == null)
      None
    else
      Some(builder.build(path, url))

  }

  private def getPath(context: Class[_], name: String) = {

    val index = context.getName().lastIndexOf('.')
    var path =
      if (index == -1) "/"
      else "/" + context.getName().substring(0, index).replace('.', '/') + "/"
    path += name

    path
  }

  trait Builder[T] {

    def build(name: String, url: URL): T
  }

  object DefaultBuilder extends Builder[ClasspathResource] {
    def build(name: String, url: URL): ClasspathResource = new ClasspathResource(name, url)
  }
}

class ClasspathResource(val name: String, val url: URL) {

  lazy val path = ClasspathResource.path + url.crc + name

  def write(event: Event[_]) {
    event.write(path)
  }

  override def toString() = name
}
