/*
 * Copyright 2011 Endea
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.http

import endea._
import endea.internal.auth._
import javax.servlet.http._

class Event[E <: Entity](
  val entity: E,
  val session: Session,
  val httpRequest: HttpServletRequest,
  val httpResponse: HttpServletResponse) {

  lazy val user = session.get(Auth.user_key)

  def attribute[T](name: String): Option[T] = {

    val value = httpRequest.getAttribute(name)
    if (value == null) None else {
      if (value.isInstanceOf[T])
        Some(value).asInstanceOf[Some[T]]
      else
        None
    }
  }

  def attribute(name: String, any: Any) {
    httpRequest.setAttribute(name, any)
  }

  def parameter(name: String) = {

    val value = httpRequest.getParameter(name)
    if (value == null || value.isEmpty()) None else Some(value)
  }

  @inline
  final def write(string: String) {
    writer.write(string)
  }

  def redirect[E <: Entity](action: HttpAction[Type[E]])(implicit manifest: Manifest[E]) {
    httpResponse.sendRedirect(Controller.encode(action)(manifest))
  }

  private lazy val writer = {
    httpResponse.setContentType("text/html; charset=UTF-8")
    httpResponse.getWriter()
  }
}