/*
 * Copyright 2011 Endea
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal

import endea._
import java.net._
import java.lang.reflect._
import scala.collection.JavaConversions._
import endea.http._
import scala.xml._
import scala.collection.mutable.ListBuffer

object Module {

  val modules: List[Module] = {

    val modules = new ListBuffer[Module]

    for (url <- classOf[Module].getClassLoader().getResources("module.xml")) {

      val xml = XML.load(url)

      val activatorNode = (xml \ "activator")
      val activator =
        if (activatorNode.length == 0)
          None
        else
          new Some(activatorNode.text)

      val entities = for (node <- (xml \\ "entity")) yield node.text

      modules += new Module(url, activator, entities)
    }

    modules.toList
  }
}

class Module private (
  val url: URL,
  val activator: Option[String],
  val entities: Seq[String]) {

  private lazy val xml = XML.load(url)

  lazy val actions: Seq[HttpAction[Entity]] = loadInstances(classOf[HttpAction[Entity]], xml \\ "action")

  private def loadInstances[T](target: Class[T], nodeSeq: NodeSeq): Seq[T] = {

    val list = new ListBuffer[T]()

    val classLoader = Thread.currentThread().getContextClassLoader()
    for (node <- nodeSeq) {
      val className = node.text
      try {
        val clazz = classLoader.loadClass(className)
        if (!target.isAssignableFrom(clazz)) {
          System.err.println("No " + target.getSimpleName() + " class: " + className)
        } else {
          val item = if (className.endsWith("$")) {
            val field = clazz.getDeclaredField("MODULE$")
            field.get(null)
          } else
            clazz.newInstance()

          list += item.asInstanceOf[T]
        }

      } catch {
        case e: ClassNotFoundException =>
          System.err.println("Class not found: " + className)
      }
    }

    list
  }

  override def toString() = url.toString()
}