/*
 * Copyright (c) 2012, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.entity

import endea._
import endea.entity._
import endea.internal._
import endea.internal.entity._
import endea.io._

import scala.collection.immutable.Stream._
import scala.collection.mutable.ArrayBuffer
import scala.math._

import com.sleepycat.je._

object Index {

  val indexDir = Endea.path \\ "index"

  lazy val environment: Environment = {

    indexDir.make()

    val envConfig = new EnvironmentConfig()
    envConfig.setAllowCreate(true)
    envConfig.setTransactional(true)

    try {
      new Environment(indexDir.jFile, envConfig)
    } catch {
      // TODO
      case e: EnvironmentLockedException => throw e
    }
  }

  def transaction = environment.beginTransaction(null, null)

  def getIndex[K, E <: Entity](attribute: String)(implicit kMf: Manifest[E], manifest: Manifest[E]): Index[K, E] = {

    MetaEntity.getByClass(manifest.erasure).findAttribute(attribute) match {
      case None => throw new Exception("Not found: " + manifest.erasure.getSimpleName() + "." + attribute)
      case Some(attribute) =>
        attribute.index match {
          case Some(index) => index.asInstanceOf[Index[K, E]]
          case None => throw new Exception("Not indexed: " + manifest.erasure.getSimpleName() + "." + attribute)
        }
    }
  }

  def stop() = {

    ClassIndex.close()

    //indexes.values.foreach(_.close)

    if (environment != null)
      environment.close()
  }

  // TODO need public by endea.test.utils
  //val indexes = new HashMap[Int, Index]

  private lazy val config = {
    val config = new DatabaseConfig()
    config.setAllowCreate(true)
    config.setSortedDuplicates(true)
    config.setTransactional(true)
    config
  }

  //
  //  private def index(transaction: Transaction, attribute: Int, entity: Entity, key: Any) {
  //
  //    getIndex(attribute).database.put(transaction, keyEntry(key), Key.encode(entity.id))
  //  }

  //  private def getIndex(attribute: Int): Index[_, _] = {
  //
  //    indexes.get(attribute) match {
  //      case Some(index) => index
  //      case None =>
  //
  //        indexes.synchronized {
  //
  //          val index = new Index(attribute.toString)
  //          indexes += attribute -> index
  //          index
  //        }
  //    }
  //  }

}

abstract class Index[K, E <: Entity](id: String)(implicit manifest: Manifest[E]) {

  private lazy val database = Index.environment.openDatabase(null, id, Index.config)

  protected def encode(key: K): DatabaseEntry

  def get(key: K, f: Stream[E] => Unit) {

    val transaction = Index.environment.beginTransaction(null, null)

    val cursor = database.openCursor(transaction, null)
    try {

      val keyEntry = encode(key)
      val foundData = new DatabaseEntry()

      if (cursor.getSearchKey(keyEntry, foundData, null) == OperationStatus.SUCCESS) {
        Get.get(toLong(foundData)) match {
          case Some(entity) =>
            f(entity #:: next(cursor, keyEntry))
          case _ => f(Empty)
        }
      } else
        f(Empty)

    } finally {
      if (cursor != null)
        cursor.close()
      if (transaction != null)
        transaction.commit()
    }

    this
  }

  private var map = Map[K, IndexedSeq[E]]()

  def get(key: K): IndexedSeq[E] = {

    map.get(key) match {
      case Some(seq) => seq
      case None =>
        synchronized {
          map.get(key) match {
            case Some(seq) => seq
            case None =>

              val seq = getSeq(key)
              map += key -> seq
              seq
          }
        }
    }
  }

  private def getSeq(key: K) = {

    var seq = IndexedSeq[E]()

    def update(stream: Stream[E]) {
      stream match {
        case Empty =>
        case x #:: xs =>
          seq :+= x
          update(xs)
      }
    }

    get(key, update(_))
    seq
  }

  def add(transaction: Transaction, key: K, entity: E) {
    database.put(transaction, encode(key), encodeEntity(entity))

    map.get(key) match {
      case None =>
      case Some(seq) =>
        map += key -> (seq :+ entity)
    }
  }

  def update(key: K, entity: E) {
  }

  def delete(transaction: Transaction, key: K, entity: E) {

    val cursor = database.openCursor(transaction, null)
    try {
      val status = cursor.getSearchBoth(encode(key), encodeEntity(entity), null)
      if (status == OperationStatus.SUCCESS)
        cursor.delete()

    } finally {
      if (cursor != null)
        cursor.close()
    }
  }

  private def close() = database.close

  private def next(cursor: Cursor, key: DatabaseEntry): Stream[E] = {

    val foundKey = new DatabaseEntry()
    val foundData = new DatabaseEntry()

    if (cursor.getNext(foundKey, foundData, null) == OperationStatus.SUCCESS &&
      foundKey.equals(key)) {
      Get.get(toLong(foundData)) match {
        case Some(entity) => entity #:: next(cursor, key)
        case None => {
          cursor.delete()
          Empty

        }
      }
    } else
      Empty
  }

  def print() {

    val cursor = database.openCursor(null, null)
    try {
      val key = new DatabaseEntry()
      val data = new DatabaseEntry()
      while (cursor.getNext(key, data, null) == OperationStatus.SUCCESS) {
        println(toLong(key) + " -> " + toLong(data))
      }
    } finally {
      if (cursor != null)
        cursor.close()
    }
  }

  protected[entity] def encodeEntity(entity: Entity) = encodeLong(entity.id)

  protected def encodeLong(value: Long) = {

    val bytes = new Array[Byte](8)
    bytes(0) = (value >>> 56).asInstanceOf[Byte]
    bytes(1) = (value >>> 48).asInstanceOf[Byte]
    bytes(2) = (value >>> 40).asInstanceOf[Byte]
    bytes(3) = (value >>> 32).asInstanceOf[Byte]
    bytes(4) = (value >>> 24).asInstanceOf[Byte]
    bytes(5) = (value >>> 16).asInstanceOf[Byte]
    bytes(6) = (value >>> 8).asInstanceOf[Byte]
    bytes(7) = value.asInstanceOf[Byte]

    new DatabaseEntry(bytes)
  }

  private[entity] def toLong(data: DatabaseEntry): Long = {

    val bytes = data.getData
    (((bytes(0) & 0xff).asInstanceOf[Long] << 56) |
      ((bytes(1) & 0xff).asInstanceOf[Long] << 48) |
      ((bytes(2) & 0xff).asInstanceOf[Long] << 40) |
      ((bytes(3) & 0xff).asInstanceOf[Long] << 32) |
      ((bytes(4) & 0xff).asInstanceOf[Long] << 24) |
      ((bytes(5) & 0xff).asInstanceOf[Long] << 16) |
      ((bytes(6) & 0xff).asInstanceOf[Long] << 8) |
      (bytes(7) & 0xff).asInstanceOf[Long])
  }
}

