/*
 * Copyright (c) 2011, 2012, Endea
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.entity

import endea._
import java.lang.reflect.Field
import java.lang.reflect.Modifier
import endea.internal._

import java.io._
import java.security.SecureRandom
import scala.collection.mutable.ArrayBuffer
import scala.collection.mutable.ListBuffer
import scala.collection.mutable.HashMap
import scala.io._

object MetaEntity {

  lazy val idField = {
    val id = classOf[Entity].getDeclaredField("id")
    id.setAccessible(true)
    id
  }

  lazy val all = {

    val entities = new ListBuffer[Class[_ <: Entity]]

    val classLoader = classOf[Module].getClassLoader()
    for (
      module <- Module.modules;
      entity <- module.entities
    ) {
      try {
        // TODO test class is assignable Class[Entity]
        entities += classLoader.loadClass(entity).asInstanceOf[Class[_ <: Entity]]
      } catch {
        case e: ClassNotFoundException =>
          System.err.println(entity + " not found - " + module)
      }
    }

    entities
  }

  /////

  private var id = 0

  private val idToMetaEntity = new HashMap[Int, MetaEntity]
  private val classToMetaEntity = new HashMap[String, MetaEntity]

  /////

  private val file = Endea.path \ "metadata"

  if (!file.exists())
    file.create()
  else {
    for (MetaEntity <- load(file jFile)) {
      idToMetaEntity.put(MetaEntity.id, MetaEntity)
      classToMetaEntity.put(MetaEntity.className, MetaEntity)
    }
  }

  /////

  def apply[E <: Entity](implicit manifest: Manifest[E]) = getByClass(manifest.erasure)

  def getByObject(any: AnyRef) = getByClass(any.getClass())

  // def getByClass[T](implicit manifest: Manifest[T]) = getByClass(manifest.erasure)

  def getByClass(clazz: Class[_]): MetaEntity = {

    val className = clazz.getName

    classToMetaEntity.get(className) match {
      case Some(meta) => meta
      case _ =>
        file.synchronized {
          classToMetaEntity.get(className) match {
            case Some(meta) => meta
            case _ =>

              val MetaEntity = make(clazz)
              save(MetaEntity, file jFile)

              idToMetaEntity.put(MetaEntity.id, MetaEntity)
              classToMetaEntity.put(className, MetaEntity)

              MetaEntity
          }
        }
    }
  }

  def getById(id: Int): Option[MetaEntity] = idToMetaEntity.get(id)

  private[internal] def make(clazz: Class[_]): MetaEntity = {

    def nextId(): Int = {
      id += 1
      id
    }

    val meta = new MetaEntity(nextId())
    meta.className = clazz.getName
    meta.dclass = clazz
    meta.dattributes =
      for (
        field <- clazz.getDeclaredFields();
        if (!Modifier.isTransient(field.getModifiers()))
      ) yield new Attribute(nextId(), field)

    meta
  }

  private[internal] def load(file: File): Seq[MetaEntity] = {

    def updateId(value: Int): Int = {
      if (id <= value)
        id = value + 1
      value
    }

    val MetaEntitys = new ArrayBuffer[MetaEntity]

    var meta: MetaEntity = null
    for (line <- Source.fromFile(file).getLines()) {
      if (line.length() > 0) {

        val trimmed = line.trim()
        val index = trimmed.indexOf(' ')

        if (trimmed.charAt(0) == '[') {
          meta = new MetaEntity(updateId(Integer.parseInt(line.substring(1, index))))
          meta.className = line.substring(index).trim
          MetaEntitys += meta

        } else {
          meta.loadedAttributes += Tuple2(
            updateId(Integer.parseInt(line.substring(0, index))),
            line.substring(index).trim())
        }
      }
    }

    MetaEntitys
  }

  private[internal] def update(meta: MetaEntity, file: File) {}

  private[internal] def save(meta: MetaEntity, file: File) {

    // val yaml = new Yaml()
    // yaml.serialize()

    def writeLine(out: BufferedWriter, id: Int, name: String) {
      out.write(Integer.toString(id))
      out.write(" ")
      out.write(name)
      out.newLine()
    }

    val out = new BufferedWriter(new FileWriter(file, true))
    try {

      out.newLine()
      out.write("[")
      writeLine(out, meta.id, meta.clazz.getName)
      out.newLine()

      for (attribute <- meta.attributes)
        writeLine(out, attribute.id, attribute.name)

      out.flush()
    } finally {
      if (out != null)
        out.close()
    }
  }
}

class MetaEntity private (val id: Int) {

  private var className: String = _

  private var loadedAttributes = new ListBuffer[(Int, String)]
  private var dclass: Class[_] = _
  private var dattributes: Seq[Attribute] = _

  lazy val clazz: Class[_] = {
    if (dclass != null)
      dclass
    else Class.forName(className)
  }

  lazy val attributes: Seq[Attribute] = {

    val attributes =
      if (dattributes != null)
        dattributes
      else {
        val attributes = new ListBuffer[Attribute]
        for ((id, name) <- loadedAttributes) {
          try {
            attributes += new Attribute(id, clazz.getDeclaredField(name))
          } catch {
            case e: NoSuchFieldException =>
              //TODO
              println("Field not found: " + clazz.getSimpleName() + "." + name)
          }
        }
        attributes.toList
      }

    attributes
  }

  private lazy val instantiator = new org.objenesis.ObjenesisStd().getInstantiatorOf(clazz)

  def newInstance(): Entity = instantiator.newInstance().asInstanceOf[Entity]

  def attribute(id: Int): Option[Attribute] = {

    for (attribute <- attributes)
      if (attribute.id == id)
        return Some(attribute)

    None
  }

  def findAttribute(name: String): Option[Attribute] = {

    for (attribute <- attributes)
      if (attribute.name == name)
        return Some(attribute)

    None
  }

  override def toString() = {

    val sb = new StringBuilder
    sb ++= clazz.getSimpleName
    sb += '['
    for (attribute <- attributes) {
      sb ++= "( "
      sb ++= attribute.id.toString
      sb ++= " -> "
      sb ++= attribute.name
      sb ++= ") "
    }
    sb += ']'

    sb.toString()
  }
}
