/*
 * Copyright (c) 2011, 2012, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.entity

import endea.Entity.Persistent
import endea.Entity.Transient
import endea.internal.entity._
import endea.Entity
import endea.entity.ClassIndex

object Put {

  def put[E <: Entity](entity: E) {

    val entities = entity :: resolveTransitiveDep(entity)
    for (e <- entities)
      putInternal(e)

    for (e <- entities) {
      e.status match {
        case Transient => EntityStore.status.set(e, Persistent)
        case _ =>
      }
    }
  }

  def putInternal[E <: Entity](entity: E) {

    new FileHandler(entity.id).write(output => Serial.write(entity, output))

    val meta = MetaEntity.getByObject(entity)

    var transaction = Index.transaction
    try {
      ClassIndex.add(transaction, entity.getClass.asInstanceOf[Class[Entity]], entity)
      for (attribute <- meta.attributes)
        attribute.index match {
          case None =>
          case Some(index) =>
            attribute.get(entity) match {
              case None => index.add(transaction, null, entity)
              case Some(value) => index.add(transaction, value, entity)
              case list: List[_] =>
                for (item <- list)
                  index.add(transaction, item, entity)
              case value => index.add(transaction, value, entity)
            }
        }
    } finally {
      if (transaction != null)
        transaction.commit()
    }

    EntityStore.cache.put(entity.id, entity)
  }

  def resolveTransitiveDep(entity: Entity): List[Entity] = {

    var entities: List[Entity] = Nil
    visit(entity)

    def visit(entity: Entity) {
      for (field <- entity.getClass().getDeclaredFields()) {
        field.setAccessible(true)
        field.get(entity) match {
          case null =>
          case Some(any) => addIfEntity(any)
          case any => addIfEntity(any)
        }
      }
    }

    def addIfEntity(any: Any) {

      if (any.isInstanceOf[Entity]) {
        val entityRef = any.asInstanceOf[Entity]
        if (entityRef.status == Transient && !entities.contains(entityRef)) {
          entities = entityRef :: entities
          visit(entityRef)
        }
      }
    }

    entities
  }
}