/*
 * Copyright (c) 2011, Endea
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.internal.entity

import endea._
import endea.io._
import endea.value._

import scala.collection.mutable.ListBuffer
import scala.collection.mutable.ArrayBuffer

object Serial {

  private val INT: Byte = 1
  private val LONG: Byte = 2
  private val FLOAT: Byte = 3
  private val DOUBLE: Byte = 4
  private val BOOLEAN: Byte = 5
  private val STRING: Byte = 6
  private val ENTITY: Byte = 7
  private val VALUE: Byte = 8
  private val LIST: Byte = 9

  final def write(entity: Entity, out: ByteOutput) {

    val metaEntity = MetaEntity.getByObject(entity)

    out.putVarInt(metaEntity.id)
    out.putVarLong(entity.id)

    /////

    for (attribute <- metaEntity.attributes) {

      val value = attribute.get(entity)
      value match {
        case null =>
        case None =>
        case s: Seq[_] if s.isEmpty =>
        case value =>
          out.putVarInt(attribute.id)
          writeValue(out, value)
      }
    }

  }

  def read(input: ByteInput): Entity = {

    val entityClassId = input.getVarInt()
    val metaEntity = MetaEntity.getById(entityClassId).
      getOrElse(throw new RuntimeException("Entity class not found: " + entityClassId))

    val entity = metaEntity.newInstance()

    MetaEntity.idField.set(entity, input.getVarLong())

    /////

    while (input.hasNext()) {

      val attributeId = input.getVarInt()
      readValue(input) match {
        case () =>
        case value =>
          metaEntity.attribute(attributeId) match {
            case Some(attribute) =>
              if (attribute.getType.isAssignableFrom(classOf[Some[_]]))
                attribute.set(entity, Some(value))
              else if (attribute.isValue) {

                if (value.isInstanceOf[List[_]]) {
                  val pType = attribute.field.getAnnotation(classOf[PType])
                  if (pType == null)
                    throw new Exception("Declare @PType to " + attribute.field)
                  val constructor = getConstructor(pType.value)
                  attribute.set(entity,
                    value.asInstanceOf[List[_]].map(item => constructor.newInstance(item.asInstanceOf[AnyRef])))
                } else {
                  val constructor = getConstructor(attribute.getType)
                  attribute.set(entity, constructor.newInstance(value.asInstanceOf[AnyRef]))
                }
              } else
                attribute.set(entity, value)
            case None =>
              throw new Exception(attributeId + "? " + MetaEntity)
          }
      }
    }

    // TODO tmp patch -> MetaEntity hava to handle null values
    for (field <- entity.getClass().getDeclaredFields()) {
      field.setAccessible(true)
      field.get(entity) match {
        case null =>
          val fieldType = field.getType()
          if (fieldType.isAssignableFrom(classOf[Option[_]])) {
            field.set(entity, None)
          } else if (fieldType.isAssignableFrom(classOf[List[_]])) {
            field.set(entity, Nil)
          }
        case _ =>
      }
    }

    entity.asInstanceOf[Entity]
  }

  private def writeValue(out: ByteOutput, value: Any) {

    value match {
      case i: Int => out.put(INT); out.putInt(i)
      case l: Long => out.put(LONG); out.putLong(l)
      case f: Float => out.put(FLOAT); out.putFloat(f)
      case d: Double => out.put(DOUBLE); out.putDouble(d)
      case b: Boolean => out.put(BOOLEAN); out.putBoolean(b)
      case s: String => out.put(STRING); out.putString(s)

      case entity: Entity =>
        out.put(ENTITY)
        out.putVarLong(entity.id)

      case value: Value[_] =>
        writeValue(out, value.asValue)

      case list: List[_] =>
        out.put(LIST)
        out.putVarInt(list.length)
        for (item <- list)
          writeValue(out, item)

      case Some(value) =>
        writeValue(out, value)

      case other => throw new Exception("Unsupported type: " + other)
    }

  }

  private def readValue(input: ByteInput): Any = {

    input.get() match {

      case INT => input.getInt().asInstanceOf[java.lang.Integer]
      case LONG => input.getLong().asInstanceOf[java.lang.Long]
      case FLOAT => input.getFloat().asInstanceOf[java.lang.Float]
      case DOUBLE => input.getDouble().asInstanceOf[java.lang.Double]
      case BOOLEAN => input.getBoolean().asInstanceOf[java.lang.Boolean]
      case STRING => input.getString()

      case ENTITY =>
        Entity.get(input.getVarLong()) match {
          case Some(entity) => entity
          case _ => // TODO deleted entity
        }

      case LIST =>
        var list = new ListBuffer[Any]()
        for (_ <- 1 to input.getVarInt())
          list += readValue(input)
        list.toList

      case other => throw new Exception("Unkown type: " + other)
    }
  }

  def getConstructor(clazz: Class[_]) = {

    val ptype = clazz.getAnnotation(classOf[PType])
    if (ptype == null)
      throw new Exception("Declare @PType to " + clazz.getName)

    clazz.getDeclaredConstructors.find(constructor => {
      val types = constructor.getParameterTypes()
      types.length == 1 && types(0) == ptype.value()
    }) match {
      case None =>
        throw new Exception("Constructor not found for: " + clazz.getName)
      case Some(constructor) =>
        constructor.setAccessible(true)
        constructor
    }
  }
}