/*
 * Copyright (c) 2012, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.io

import java.nio.channels.SeekableByteChannel

class ByteInput private[io] (channel: SeekableByteChannel) {

  private val buffer = java.nio.ByteBuffer.allocate(8192)
  private var bytesRead = 0

  read()

  def hasNext(): Boolean = {

    if (bytesRead == -1)
      false
    else {
      if (buffer.remaining > 0)
        true
      else {
        read()
        hasNext()
      }
    }
  }

  def get(): Byte = {

    if (!buffer.hasRemaining())
      read()
    buffer.get()
  }

  def getInt(): Int = {
    (getByteAsInt() << 24) |
      (getByteAsInt() << 16) |
      (getByteAsInt() << 8) |
      (getByteAsInt())
  }

  def getVarInt(): Int = {

    var int = buffer.get().asInstanceOf[Int]
    if ((int & 0x80) == 0)
      return int

    int = int & 0x7f
    var shift = 7
    while (shift < 32) {
      val byte = buffer.get().asInstanceOf[Int]
      int |= (byte & 0x7F) << shift
      if ((byte & 0x80) == 0)
        return int
      shift += 7
    }

    throw new Exception("Malformed var int")
  }

  def getLong(): Long = {

    (getByteAsLong() << 56) |
      (getByteAsLong() << 48) |
      (getByteAsLong() << 40) |
      (getByteAsLong() << 32) |
      (getByteAsLong() << 24) |
      (getByteAsLong() << 16) |
      (getByteAsLong() << 8) |
      getByteAsLong()
  }

  def getVarLong(): Long = {

    var long = 0L

    var shift = 0
    while (shift < 64) {
      val byte = getByteAsLong()
      long |= (byte & 0x7F) << shift
      if ((byte & 0x80) == 0)
        return long
      shift += 7
    }

    throw new Exception("Malformed var long")
  }

  def getFloat(): Float = java.lang.Float.intBitsToFloat(getInt())

  def getDouble(): Double = java.lang.Double.longBitsToDouble(getLong())

  def getBoolean(): Boolean = buffer.get() != 0

  def getString(): String = {

    val chars = new Array[Char](getVarInt())

    val len = getVarInt()
    var bi = 0
    var ci = 0
    while (bi < len) {

      val char1 = get()
      val test = ((char1 & 0xff) >> 4)

      if (test < 8) {
        chars(ci) = char1.asInstanceOf[Char];
        bi += 1
      } else if (test == 12 || test == 13) {
        chars(ci) = (((char1 & 0x1F) << 6) | (get() & 0x3F)).asInstanceOf[Char]
        bi += 2
      } else if (test == 14) {
        chars(ci) = (((char1 & 0x0F) << 12) | ((get() & 0x3F) << 6) | ((get() & 0x3F) << 0)).asInstanceOf[Char]
        bi += 3
      } else
        throw new IllegalArgumentException()
      ci += 1
    }

    new String(chars)
  }

  private[io] def close() = channel.close()

  private def read() {

    if (bytesRead != -1) {
      buffer.clear()
      bytesRead = channel.read(buffer)
      buffer.flip()
    }
  }

  @inline
  private def getByteAsInt(): Int = (get() & 0xff).asInstanceOf[Int]

  @inline
  private def getByteAsLong(): Long = (get() & 0xff).asInstanceOf[Long]
}