/*
 * Copyright (c) 2012, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.io

import java.nio.channels.SeekableByteChannel
import java.nio.charset.Charset

object ByteOutput {

  val charSet = Charset.forName("UTF-8")
}

class ByteOutput private[io] (channel: SeekableByteChannel) {

  private val buffer = java.nio.ByteBuffer.allocate(8192)
  buffer.clear()

  def put(byte: Byte) {

    if (buffer.remaining == 0)
      write()

    buffer.put(byte)
  }

  def putInt(int: Int) {

    putByte(int >>> 24)
    putByte(int >>> 16)
    putByte(int >>> 8)
    putByte(int)
  }

  def putVarInt(int: Int) {

    var intCopy = int
    while (true) {
      if ((intCopy & ~0x7F) == 0) {
        putByte(intCopy)
        return
      } else {
        putByte((intCopy & 0x7F) | 0x80)
        intCopy >>>= 7
      }
    }
  }

  def putLong(long: Long) {

    putByte(long >>> 56)
    putByte(long >>> 48)
    putByte(long >>> 40)
    putByte(long >>> 32)
    putByte(long >>> 24)
    putByte(long >>> 16)
    putByte(long >>> 8)
    putByte(long)
  }

  def putVarLong(long: Long) {

    var longCopy = long
    while (true) {
      if ((longCopy & ~0x7FL) == 0) {
        putByte(longCopy)
        return
      } else {
        putByte((longCopy & 0x7F) | 0x80)
        longCopy >>>= 7
      }
    }
  }

  def putFloat(float: Float) {
    putInt(java.lang.Float.floatToIntBits(float))
  }

  def putDouble(double: Double) {
    putLong(java.lang.Double.doubleToLongBits(double))
  }

  def putBoolean(boolean: Boolean) {
    putByte(if (boolean) 1 else 0)
  }

  def putString(string: String) {

    val chars = string.toCharArray()
    putVarInt(chars.length)

    var len = 0
    for (c <- chars) {
      if ((c >= 0x0001) && (c <= 0x007F))
        len += 1
      else if (c > 0x07FF)
        len += 3
      else
        len += 2
    }
    putVarInt(len)

    for (char <- chars) {
      if ((char >= 0x0001) && (char <= 0x007F)) {
        putByte(char)
      } else if (char > 0x07FF) {
        putByte(0xE0 | ((char >> 12) & 0x0F))
        putByte(0x80 | ((char >> 6) & 0x3F))
        putByte(0x80 | ((char >> 0) & 0x3F))
      } else {
        putByte(0xC0 | ((char >> 6) & 0x1F))
        putByte(0x80 | ((char >> 0) & 0x3F))
      }
    }
  }

  private[io] def close() {

    try {
      write()
    } finally {
      channel.close()
    }
  }

  private def write() {

    buffer.flip()
    while (buffer.hasRemaining())
      channel.write(buffer)
    buffer.clear()
  }

  @inline
  private def putByte(int: Int) = put(int.asInstanceOf[Byte])

  @inline
  private def putByte(long: Long) = put(long.asInstanceOf[Byte])

}
