/*
 * Copyright (c) 2012, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.io

import java.io.{ File => JFile, _ }
import java.nio.file.{ Path => JPath, _ }
import java.io.IOException
import java.nio.file.attribute.BasicFileAttributes

object Directory {

  val userDir = new Directory(new JFile(System.getProperty("user.dir")))

  def -\\() = userDir

  def -\\(name: String) = new Directory(new JFile(userDir jFile, name))

  /**
   * Root directory
   */
  def :\\() {}

  /**
   * Home directory
   */
  def ~\\() {}
}

class Directory private[io] (jFile: JFile) extends Path(jFile) {

  /** Directory **/
  def \\(name: String) = new Directory(new JFile(jFile, name))

  /** File **/
  def \(name: String) = new File(new JFile(jFile, name))

  /** Temp file */
  def %\ = {
    jFile.mkdirs
    new File(Files.createTempFile(jFile.toPath, null, null).toFile())
  }

  /** Link **/
  def >\(name: String) = new Link(new JFile(jFile, name))

  def ?\(name: String): Path = {

    val file = new JFile(jFile, name)
    if (file.isDirectory)
      new Directory(file)
    else
      new File(file)
  }

  def tmp = {
    if (!jFile.exists())
      jFile.mkdirs()
    new File(Files.createTempFile(jFile.toPath(), null, null).toFile())
  }

  def make(): Directory = {
    jFile.mkdirs()
    this
  }

  def isEmpty() = !Files.newDirectoryStream(jFile.toPath).iterator.hasNext

  def list(): Array[String] = if (jFile.exists) jFile.list else Array.empty

  def _list() = Files.newDirectoryStream(jFile.toPath)

  def delete() {

    if (jFile.exists()) {
      Files.walkFileTree(jFile.toPath(), new SimpleFileVisitor[JPath]() {

        override def visitFile(file: JPath, attrs: BasicFileAttributes): FileVisitResult = {
          Files.delete(file)
          return FileVisitResult.CONTINUE
        }

        override def postVisitDirectory(dir: JPath, exception: IOException): FileVisitResult = {
          if (exception == null) {
            Files.delete(dir)
            return FileVisitResult.CONTINUE
          } else
            throw exception
        }
      })
    }
  }

  override def toString() = jFile.toString + "/"
}