/*
 * Copyright (c) 2012, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.io

import java.io.{ File => JFile, _ }
import java.nio.file._
import java.security.MessageDigest
import endea.io.io._
import java.nio.channels.AsynchronousFileChannel

object File {

  def %\ = new File(Files.createTempFile(null, null).toFile())
}

class File private[io] (jFile: JFile) extends Path(jFile) {

  lazy val parent = new Directory(jFile.getParentFile)

  def write(f: ByteOutput => Unit) {

    create()
    val output = new ByteOutput(new FileOutputStream(jFile).getChannel())
    try {
      f(output)
    } finally {
      output.close()
    }
  }

  def read(f: ByteInput => Unit) {

    val input = new ByteInput(new FileInputStream(jFile).getChannel())
    try {
      f(input)
    } finally {
      input.close()
    }
  }

  def create() {
    if (!jFile.exists) {
      jFile.getParentFile.mkdirs()
      jFile.createNewFile
    }
  }

  def delete() = jFile.delete()

  def contentType(): Option[String] = Files.probeContentType(jFile.toPath()) match {
    case null => None
    case ct => Some(ct)
  }

  def md5() = {

    val buffer = new Array[Byte](1024)
    val md5 = MessageDigest.getInstance("MD5")

    val input = new FileInputStream(jFile)
    try {
      Stream.continually(input.read(buffer)).
        takeWhile(_ != -1).
        foreach(md5.update(buffer, 0, _))
    } finally {
      if (input != null)
        input.close()
    }

    // TODO use an array
    new java.math.BigInteger(1, md5.digest()).toString(16)
  }

  def moveTo(that: File, replaceExisting: Boolean = false, atomicMove: Boolean = false) {

    that.mkDirs()
    Files.move(jFile.toPath(), that.jFile.toPath())
  }

  def toInputStream(): FileInputStream = new FileInputStream(jFile)

  def toOutputStream(): FileOutputStream = {
    if (!jFile.exists)
      mkDirs()
    new FileOutputStream(jFile)
  }

  def copyTo(outputStream: OutputStream) {
    toInputStream().copyTo(outputStream)
  }

  //  def printToFile(f: java.io.File)(op: java.io.PrintWriter => Unit) {
  //    val p = new java.io.PrintWriter(f)
  //    try { op(p) } finally { p.close() }
  //  }

  override def toString = jFile.toString()

  private def mkDirs() {

    val parent = jFile.getParentFile()
    if (parent != null)
      parent.mkdirs()
  }
}