/*
 * Copyright (c) 2012, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.io

import endea.internal.io.Utils

import java.io.Closeable
import java.io.InputStream
import java.util.ArrayList

import org.apache.http.client.entity.UrlEncodedFormEntity
import org.apache.http.client.methods.HttpGet
import org.apache.http.client.methods.HttpPost
import org.apache.http.client.methods.HttpRequestBase
import org.apache.http.client.utils.URIUtils
import org.apache.http.client.utils.URLEncodedUtils
import org.apache.http.impl.client.DefaultHttpClient
import org.apache.http.message.BasicNameValuePair
import org.apache.http.protocol.HTTP
import org.apache.http.HttpResponse
import org.apache.http.NameValuePair

import org.jsoup._
import org.jsoup.nodes._

object Http {

  def apply(host: String) = new Http(host)
}

class Http(val host: String) extends Closeable {

  private val client = new DefaultHttpClient()

  def get(path: String, query: Traversable[(String, String)] = Nil) = new Get(path, query)

  def post(path: String, query: Traversable[(String, String)] = Nil) = new Post(path, query)

  override def close() {
    client.getConnectionManager().shutdown();
  }

  abstract class Request() {

    private[io] def request: HttpRequestBase

    //def execute() = new Response(client.execute(request))

    def execute(f: InputStream => Unit): Response = {

      val httpResponse = client.execute(request)

      val entity = httpResponse.getEntity()
      if (entity != null && entity.isStreaming()) {
        val inputStream = entity.getContent()
        try {
          f(inputStream)
        } finally {
          Utils.close(inputStream)
        }
      }

      new Response(httpResponse)
    }

    def document(): Document = {

      var document: Document = null
      execute(input => document = Jsoup.parse(input, null, ""))
      document
    }

    def print() = {
      println(request.getRequestLine())
      this
    }

    private[io] def format(query: Traversable[(String, String)]): String = {

      if (query.isEmpty)
        null
      else
        URLEncodedUtils.format(pairs(query), HTTP.UTF_8)
    }

    private[io] def pairs(pairs: Traversable[(String, String)]) = {

      val list = new ArrayList[NameValuePair]()
      for ((name, value) <- pairs)
        list.add(new BasicNameValuePair(name, value))
      list
    }
  }

  class Get(path: String, query: Traversable[(String, String)]) extends Request {

    val request = new HttpGet(
      URIUtils.createURI("http", host, -1, path, format(query), null))
  }

  class Post(path: String, query: Traversable[(String, String)]) extends Request {

    val request = new HttpPost(URIUtils.createURI("http", host, -1, path, format(query), null))

    def data(data: Traversable[(String, String)]): Post = {
      request.setEntity(new UrlEncodedFormEntity(pairs(data), HTTP.UTF_8))
      this
    }
  }

  class Response(response: HttpResponse) {

    def print() {
      println(response.getStatusLine().toString())
      for (header <- response.getAllHeaders()) {
        println(header.getName() + ": " + header.getValue())
      }
    }
  }

  //  class Response(response: HttpResponse) {
  //
  //    lazy val headers = response.getAllHeaders()
  //
  //    def handle(f: InputStream => Unit) {
  //
  //      val entity = response.getEntity()
  //
  //      if (entity != null && entity.isStreaming()) {
  //        val inputStream = entity.getContent()
  //        try {
  //          f(inputStream)
  //        } finally {
  //          Utils.close(inputStream)
  //        }
  //      }
  //    }
  //  }

}