/*
 * Copyright (c) 2011, Endea.org
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.value

import Money._
import Money.Currency._

object Money {

  def apply(amount: Long, currency: Currency) = {
    new Money(amount, currency)
  }

  def apply(amount: Double, currency: Currency) = {
    new Money(amount, currency)
  }

  implicit def doubleConverter(amount: Double) = new {

    def eur = {
      new Money(amount, EUR)
    }

    def usd = {
      new Money(amount, USD)
    }
  }

  // ISO_4217
  sealed abstract class Currency(code: String, name: String, cents: Boolean = true) {

    def cents(amount: Double): Long = {

      if (cents)
        (amount * 100).toLong
      else
        amount.toLong
    }

    def toString(amount: Long) = {

      if (cents)
        (amount / 100) + "." + (amount % 100) + " " + code
      else
        amount + " " + code
    }
  }

  case object EUR extends Currency("EUR", "Euro", true)
  case object USD extends Currency("USD", "United States dollar", true)
  case object JPY extends Currency("JPY", "Japanese yen")
}

class Money[C <: Currency](val amount: Long, currency: C) extends Ordered[Money[C]] with Value[String] {

  def this(amount: Double, currency: C) = this(currency.cents(amount), currency)

  // private def this(string: String) = new Money(34L, Currency.USD)

  def +(that: Money[C]) = new Money(amount + that.amount, currency)
  def -(that: Money[C]) = new Money(amount - that.amount, currency)

  override def compare(that: Money[C]) = (amount - that.amount).toInt

  lazy val asValue = currency + amount.toString

  override def toString = currency.toString(amount)

  override def equals(that: Any) =
    that match {
      case m: Money[C] => amount == m.amount // TODO && currency == m.currency
      case _ => false
    }

}

