/*
 * Copyright 2011 Endea
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package endea.value

import endea._

import java.text._
import java.util._

object Time {

  def apply(millis: Long) = new Time(millis)

  def now = new Time(System.currentTimeMillis())
}

@PType(classOf[Long])
class Time private (
  /** The milliseconds from 1970-01-01T00:00:00Z */
  val millis: Long) extends Ordered[Time] with Value[Long] {

  def -(time: Time): Time = {
    new Time(time.millis - millis)
  }

  def |(end: Time) = new TimeInterval(this, end)

  /* http://download.oracle.com/javase/1.4.2/docs/api/java/text/SimpleDateFormat.html */
  def format(pattern: String) = {

    val format = new SimpleDateFormat(pattern)
    format.format(new java.util.Date(millis))
  }

  def asValue = millis

  override def compare(that: Time) = (millis - that.millis).toInt

  override def equals(that: Any) = that match {
    case time: Time => millis == time.millis
    case _ => false
  }

  override def hashCode = millis.asInstanceOf[Int]

  override def toString() = format("EEE, d MMM yyyy HH:mm:ss Z")
}

