/*- Package Declaration ------------------------------------------------------*/

package org.epics.ca.impl.monitor.blockingqueue;

/*- Imported packages --------------------------------------------------------*/

/*- Interface Declaration ----------------------------------------------------*/
/*- Class Declaration --------------------------------------------------------*/

import net.jcip.annotations.ThreadSafe;
import org.apache.commons.lang3.Validate;
import org.epics.ca.impl.TypeSupports;
import org.epics.ca.impl.monitor.MonitorNotificationService;
import org.epics.ca.util.logging.LibraryLogManager;

import java.nio.ByteBuffer;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.logging.Logger;

@ThreadSafe
public class BlockingQueueMonitorNotificationService<T> implements MonitorNotificationService<T>, Supplier<T>
{

/*- Public attributes --------------------------------------------------------*/
/*- Private attributes -------------------------------------------------------*/

   private static final Logger logger = LibraryLogManager.getLogger( BlockingQueueMonitorNotificationService.class );

   private final ThreadPoolExecutor executor;
   private final Consumer<? super T> consumer;
   private final BlockingQueue<T> valueQueue;

   private T deserializedValue;


/*- Main ---------------------------------------------------------------------*/
/*- Constructor --------------------------------------------------------------*/

   /**
    * Creates a new monitor notifier based on a standard Java BlockingQueue
    * and a ThreadPoolExecutor.
    *
    * The supplied executor should be appropriately configured according to
    * the needs of the service. For example it should leverage off the
    * appropriate type of BlockingQueue (bounded or unbounded) and support
    * the appropriate number of threads (single or multiple).
    *
    * Similarly the supplied queue should be appropriately configured (it
    * could be bounded or unbounded).
    *
    * @param executor the executor.
    * @param valueQueue the consumer's value notification queue.
    * @param consumer the consumer to whom published values will be sent.
    *
    * @throws NullPointerException if the executor was null.
    * @throws NullPointerException if the consumer was null.
    */
   BlockingQueueMonitorNotificationService( ThreadPoolExecutor executor, BlockingQueue<T> valueQueue, Consumer<? super T> consumer  )
   {
      this.executor = Validate.notNull( executor );
      this.valueQueue = Validate.notNull( valueQueue );
      this.consumer = Validate.notNull( consumer );

      this.deserializedValue = null;
    }

/*- Public methods -----------------------------------------------------------*/

   /**
    * {@inheritDoc}
    *
    * @implNote
    * This implementation does not accept null as a valid publication value.
    *
    * @throws NullPointerException if the passed value was null.
    */
   @Override
   public boolean publish( ByteBuffer dataBuffer, TypeSupports.TypeSupport<T> typeSupport, int dataCount )
   {
      // The deserializer is optimised to reuse the same data structure thus
      // avoiding the cost of object creation
      deserializedValue = typeSupport.deserialize( dataBuffer, deserializedValue , dataCount );
      return publish( deserializedValue );
   }

   /**
    * {@inheritDoc}
    *
    * @implNote
    * This implementation does not accept null as a publication value.
    *
    * @throws NullPointerException if the passed value was null.
    */
   @Override
   synchronized public boolean publish( T value )
   {
      Validate.notNull( value );

      // Add the latest value to the tail of the notification queue, where necessary evicting
      // the oldest value to ensure success.
      boolean overrun = false;
      if ( ! valueQueue.offer( value ) )
      {
         logger.finest( String.format( "Buffer is full [size is: %d]", valueQueue.size() ) );
         overrun = true;
         final T discardedValue = valueQueue.remove();
         logger.finest( String.format( "Removing and throwing away oldest queue item, %s", discardedValue ) );

         // Theoretically this call could throw an IllegalStateException but it should
         // not do so since the previous remove operation should now guarantee success.
         valueQueue.add( value );
      }
      else
      {
         logger.finest( String.format("Added new item to buffer [size is: %d]", valueQueue.size() ) );

         // In the case that there is a new notifcation item in the queue create a new task to pass the
         // value on to the consumer.
         logger.finest( String.format( "Queueing Task for consumer '%s' on work queue '%s'. Latest value is: '%s'", consumer, executor.getQueue().hashCode(), value ) );
         executor.submit( new MonitorNotificationTask<>( consumer, this ) );
      }

      // Return true for success; false if there was a buffer overrun.
      return ! overrun;
   }

   /**
    * {@inheritDoc}
    *
    * @implNote
    * This implementation returns the oldest value from the notification queue.
    */
   @Override
   synchronized public T get()
   {
      // Check the precondition has not been violated. If it has there is a programming error
      Validate.isTrue( ! valueQueue.isEmpty(), "programming error - value notification queue was unexpectedly empty" );

      final T value = valueQueue.remove();
      logger.finest(  String.format( "Retrieved value '%s'", value ) );

      // Get the oldest value from the head of the notification value queue
      return value;
   }

   /**
    * {@inheritDoc}
    * <p>
    * The implementation here does not need to do anything since the service leverages
    * off a shared ThreadPoolExecutor whose lifecycle is managed outside the scope of
    * this object's lifetime.
    */
   @Override public void init() { }

   /**
    * {@inheritDoc}
    * <p>
    * The implementation here does not need to do anything since the service leverages
    * off a shared ThreadPoolExecutor whose lifecycle is managed outside the scope of
    * this object's lifetime.
    */
   @Override
   public void close()
   {
      logger.finest( "Closing monitor notification service for consumer." );
   }

/*- Private methods ----------------------------------------------------------*/
/*- Nested Classes -----------------------------------------------------------*/

}
