/*
 * Decompiled with CFR 0.152.
 */
package org.epics.ca.impl.search;

import java.nio.ByteBuffer;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Logger;
import org.epics.ca.impl.ChannelImpl;
import org.epics.ca.impl.Messages;
import org.epics.ca.impl.UdpBroadcastTransport;
import org.epics.ca.impl.search.SearchTimer;
import org.epics.ca.util.logging.LibraryLogManager;

public class ChannelSearchManager {
    private static final int MIN_SEND_INTERVAL_MS_DEFAULT = 100;
    private static final int MAX_SEND_INTERVAL_MS_DEFAULT = 30000;
    private static final int INTERVAL_MULTIPLIER_DEFAULT = 2;
    private static final int MESSAGE_COALESCENCE_TIME_MS = 3;
    private static final int MAX_NUMBER_IMMEDIATE_PACKETS = 5;
    private static final int IMMEDIATE_PACKETS_DELAY_MS = 10;
    private static final Logger logger = LibraryLogManager.getLogger(ChannelSearchManager.class);
    private final long minSendInterval;
    private final long maxSendInterval;
    private final int intervalMultiplier;
    private final SearchTimer timer = new SearchTimer();
    private final AtomicBoolean canceled = new AtomicBoolean();
    private final AtomicInteger immediatePacketCount = new AtomicInteger();
    private final AtomicInteger channelCount = new AtomicInteger();
    private final UdpBroadcastTransport udpBroadcastTransport;
    private final AtomicInteger sequenceNumber = new AtomicInteger(0);
    private final ByteBuffer sendBuffer;

    public ChannelSearchManager(UdpBroadcastTransport udpBroadcastTransport) {
        this.udpBroadcastTransport = udpBroadcastTransport;
        this.minSendInterval = 100L;
        this.maxSendInterval = 30000L;
        this.intervalMultiplier = 2;
        this.sendBuffer = ByteBuffer.allocateDirect(1024);
        this.initializeSendBuffer();
    }

    public boolean registerChannel(ChannelImpl<?> channel) {
        if (this.canceled.get()) {
            return false;
        }
        ChannelSearchTimerTask timerTask = new ChannelSearchTimerTask(channel);
        channel.setTimerId(timerTask);
        this.timer.executeAfterDelay(3L, timerTask);
        this.channelCount.incrementAndGet();
        return true;
    }

    public void unregisterChannel(ChannelImpl<?> channel) {
        if (this.canceled.get()) {
            return;
        }
        Object timerTask = channel.getTimerId();
        if (timerTask != null) {
            SearchTimer.cancel(timerTask);
            channel.setTimerId(null);
        }
        this.channelCount.decrementAndGet();
    }

    public int registeredChannelCount() {
        return this.channelCount.get();
    }

    public void beaconAnomalyNotify() {
        logger.fine("A beacon anomaly has been detected.");
        if (this.canceled.get()) {
            return;
        }
        logger.fine("Reinstigating channel search.");
        this.timer.rescheduleAllAfterDelay(0L);
    }

    public void cancel() {
        if (this.canceled.getAndSet(true)) {
            return;
        }
        this.timer.shutDown();
    }

    private void initializeSendBuffer() {
        this.sendBuffer.clear();
        this.sequenceNumber.incrementAndGet();
        Messages.generateVersionRequestMessage(this.udpBroadcastTransport, this.sendBuffer, (short)0, this.sequenceNumber.get(), true);
    }

    private synchronized void flushSendBuffer() {
        if (this.immediatePacketCount.incrementAndGet() >= 5) {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            this.immediatePacketCount.set(0);
        }
        this.udpBroadcastTransport.send(this.sendBuffer);
        this.initializeSendBuffer();
    }

    private synchronized boolean generateSearchRequestMessage(ChannelImpl<?> channel, boolean allowNewFrame) {
        boolean success = channel.generateSearchRequestMessage(this.udpBroadcastTransport, this.sendBuffer);
        if (!success) {
            this.flushSendBuffer();
            if (allowNewFrame) {
                channel.generateSearchRequestMessage(this.udpBroadcastTransport, this.sendBuffer);
            }
            return true;
        }
        return false;
    }

    public void searchResponse(ChannelImpl<?> channel) {
        this.unregisterChannel(channel);
    }

    private class ChannelSearchTimerTask
    extends SearchTimer.TimerTask {
        private final ChannelImpl<?> channel;

        ChannelSearchTimerTask(ChannelImpl<?> channel) {
            this.channel = channel;
        }

        @Override
        public long timeout() {
            ChannelSearchManager.this.generateSearchRequestMessage(this.channel, true);
            if (!ChannelSearchManager.this.timer.hasNext(3L)) {
                ChannelSearchManager.this.flushSendBuffer();
                ChannelSearchManager.this.immediatePacketCount.set(0);
            }
            long dT = this.getDelay();
            if ((dT *= (long)ChannelSearchManager.this.intervalMultiplier) > ChannelSearchManager.this.maxSendInterval) {
                dT = ChannelSearchManager.this.maxSendInterval;
            }
            if (dT < ChannelSearchManager.this.minSendInterval) {
                dT = ChannelSearchManager.this.minSendInterval;
            }
            return dT;
        }
    }
}

