/*
 * Decompiled with CFR 0.152.
 */
package org.epics.ca.impl.repeater;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.epics.ca.ThreadWatcher;
import org.epics.ca.impl.JavaProcessManager;
import org.epics.ca.impl.repeater.NetworkUtilities;
import org.epics.ca.impl.repeater.UdpSocketReserver;
import org.epics.ca.impl.repeater.UdpSocketUtilities;
import org.epics.ca.util.logging.LibraryLogManager;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.hamcrest.core.Is;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.RepeatedTest;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;

class UdpSocketUtilitiesTest {
    private static final Logger logger = LibraryLogManager.getLogger(UdpSocketUtilitiesTest.class);
    private ThreadWatcher threadWatcher;

    UdpSocketUtilitiesTest() {
    }

    @BeforeAll
    static void beforeAll() {
        MatcherAssert.assertThat(NetworkUtilities.verifyTargetPlatformNetworkStackIsChannelAccessCompatible(), Is.is(true));
    }

    @BeforeEach
    void beforeEach() {
        this.threadWatcher = ThreadWatcher.start();
    }

    @AfterEach
    void afterEach() {
        this.threadWatcher.verify();
    }

    @Test
    void testInetAddress_getAllByName() throws UnknownHostException {
        MatcherAssert.assertThat(Arrays.toString(InetAddress.getAllByName("0.0.0.0")), Is.is("[/0.0.0.0]"));
        MatcherAssert.assertThat(Arrays.toString(InetAddress.getAllByName(null)), Is.is("[localhost/127.0.0.1]"));
        InetAddress[] localInetAddresses = InetAddress.getAllByName("localhost");
        Arrays.stream(localInetAddresses).map(InetAddress::toString).forEach(i2 -> MatcherAssert.assertThat(i2, Matchers.anyOf(Is.is("localhost/127.0.0.1"), Is.is("localhost/127.0.1.1"), Is.is("localhost/0:0:0:0:0:0:0:1"))));
    }

    @Test
    void testInetAddress_getLoopbackAddress_isNotAnyLocalAddress() {
        MatcherAssert.assertThat(InetAddress.getLoopbackAddress().isAnyLocalAddress(), Is.is(false));
    }

    @Test
    void testInetAddress_getLoopbackAddress_is_as_expected() {
        MatcherAssert.assertThat(InetAddress.getLoopbackAddress().toString(), Is.is("localhost/127.0.0.1"));
    }

    @Test
    void testInetAddress_getLocalHost_isNotAnyLocalAddress() throws UnknownHostException {
        MatcherAssert.assertThat(InetAddress.getLocalHost().isAnyLocalAddress(), Is.is(false));
    }

    @Test
    void testInetAddress_getLocalHostAddress_and_getLoopbackAddress() throws UnknownHostException {
        logger.info("The localhost address is: " + InetAddress.getLocalHost());
        logger.info("The loopback address is: " + InetAddress.getLoopbackAddress());
    }

    @Test
    void testInetAddress_getByName_IP_0_0_0_0_isAnyLocalAddress() throws UnknownHostException {
        MatcherAssert.assertThat(InetAddress.getByName("0.0.0.0").isAnyLocalAddress(), Is.is(true));
    }

    @Test
    void testInetSocketAddresss_constructor_withWildcardAddressAndDefinedPort() throws UnknownHostException {
        InetSocketAddress wildcardAddress = new InetSocketAddress(1234);
        MatcherAssert.assertThat(wildcardAddress.toString(), Is.is("0.0.0.0/0.0.0.0:1234"));
        MatcherAssert.assertThat(wildcardAddress.isUnresolved(), Is.is(false));
        MatcherAssert.assertThat(wildcardAddress.getPort(), Is.is(1234));
        MatcherAssert.assertThat(wildcardAddress.getAddress(), Is.is(InetAddress.getByName("0.0.0.0")));
        MatcherAssert.assertThat(wildcardAddress.getAddress().toString(), Is.is("0.0.0.0/0.0.0.0"));
        MatcherAssert.assertThat(wildcardAddress.getHostName(), Is.is("0.0.0.0"));
        MatcherAssert.assertThat(wildcardAddress.getHostString(), Is.is("0.0.0.0"));
    }

    @Test
    void testInetSocketAddresss_constructor_withWildcardAddressAndEphemeralPort() throws UnknownHostException {
        InetSocketAddress ephemeralAddress = new InetSocketAddress(0);
        MatcherAssert.assertThat(ephemeralAddress.toString(), Is.is("0.0.0.0/0.0.0.0:0"));
        MatcherAssert.assertThat(ephemeralAddress.isUnresolved(), Is.is(false));
        MatcherAssert.assertThat(ephemeralAddress.getPort(), Is.is(0));
        MatcherAssert.assertThat(ephemeralAddress.getAddress(), Is.is(InetAddress.getByName("0.0.0.0")));
        MatcherAssert.assertThat(ephemeralAddress.getAddress().toString(), Is.is("0.0.0.0/0.0.0.0"));
        MatcherAssert.assertThat(ephemeralAddress.getHostName(), Is.is("0.0.0.0"));
        MatcherAssert.assertThat(ephemeralAddress.getHostString(), Is.is("0.0.0.0"));
    }

    @Test
    void testInetSocketAddresss_constructor_withEmptyHostnameString() {
        InetSocketAddress emptyHostnameAddress = new InetSocketAddress("", 1234);
        MatcherAssert.assertThat(emptyHostnameAddress.toString(), Is.is("localhost/127.0.0.1:1234"));
        MatcherAssert.assertThat(emptyHostnameAddress.isUnresolved(), Is.is(false));
        MatcherAssert.assertThat(emptyHostnameAddress.getPort(), Is.is(1234));
        MatcherAssert.assertThat(emptyHostnameAddress.getAddress(), Is.is(InetAddress.getLoopbackAddress()));
        MatcherAssert.assertThat(emptyHostnameAddress.getAddress().toString(), Is.is("localhost/127.0.0.1"));
        MatcherAssert.assertThat(emptyHostnameAddress.getHostName(), Is.is("localhost"));
        MatcherAssert.assertThat(emptyHostnameAddress.getHostString(), Is.is("localhost"));
    }

    @Test
    void testInetSocketAddresss_constructor_withHostnameStringSetToLocalhost() {
        InetSocketAddress loopbackAddress = new InetSocketAddress("localhost", 1234);
        MatcherAssert.assertThat(loopbackAddress.toString(), Is.is("localhost/127.0.0.1:1234"));
        MatcherAssert.assertThat(loopbackAddress.isUnresolved(), Is.is(false));
        MatcherAssert.assertThat(loopbackAddress.getPort(), Is.is(1234));
        MatcherAssert.assertThat(loopbackAddress.getAddress(), Is.is(InetAddress.getLoopbackAddress()));
        MatcherAssert.assertThat(loopbackAddress.getAddress().toString(), Is.is("localhost/127.0.0.1"));
        MatcherAssert.assertThat(loopbackAddress.getHostName(), Is.is("localhost"));
        MatcherAssert.assertThat(loopbackAddress.getHostString(), Is.is("localhost"));
    }

    @Test
    void testInetSocketAddresss_constructor_withHostnameStringSetToResolvableHost() throws UnknownHostException {
        InetSocketAddress resolvableHostAddress = new InetSocketAddress("psi.ch", 1234);
        MatcherAssert.assertThat(resolvableHostAddress.isUnresolved(), Is.is(false));
        MatcherAssert.assertThat(resolvableHostAddress.toString(), Is.is("psi.ch/192.33.120.32:1234"));
        MatcherAssert.assertThat(resolvableHostAddress.getPort(), Is.is(1234));
        MatcherAssert.assertThat(resolvableHostAddress.getAddress(), Is.is(InetAddress.getByName("psi.ch")));
        MatcherAssert.assertThat(resolvableHostAddress.getAddress().toString(), Is.is("psi.ch/192.33.120.32"));
        MatcherAssert.assertThat(resolvableHostAddress.getHostName(), Is.is("psi.ch"));
        MatcherAssert.assertThat(resolvableHostAddress.getHostString(), Is.is("psi.ch"));
    }

    @Test
    void testInetSocketAddresss_constructor_withNullHostnameString() throws UnknownHostException {
        InetSocketAddress nullHostAddress = new InetSocketAddress((InetAddress)null, 1234);
        MatcherAssert.assertThat(nullHostAddress.toString(), Is.is("0.0.0.0/0.0.0.0:1234"));
        MatcherAssert.assertThat(nullHostAddress.isUnresolved(), Is.is(false));
        MatcherAssert.assertThat(nullHostAddress.getPort(), Is.is(1234));
        MatcherAssert.assertThat(nullHostAddress.getAddress(), Is.is(InetAddress.getByName("0.0.0.0")));
        MatcherAssert.assertThat(nullHostAddress.getAddress().toString(), Is.is("0.0.0.0/0.0.0.0"));
        MatcherAssert.assertThat(nullHostAddress.getHostName(), Is.is("0.0.0.0"));
        MatcherAssert.assertThat(nullHostAddress.getHostString(), Is.is("0.0.0.0"));
    }

    @Test
    void testInetSocketAddresss_constructor_withUnresolvedHost() {
        InetSocketAddress unresolvableHostAddress = InetSocketAddress.createUnresolved("somehost", 1234);
        MatcherAssert.assertThat(unresolvableHostAddress.toString(), Is.is("somehost:1234"));
        MatcherAssert.assertThat(unresolvableHostAddress.isUnresolved(), Is.is(true));
        MatcherAssert.assertThat(unresolvableHostAddress.getPort(), Is.is(1234));
        MatcherAssert.assertThat(unresolvableHostAddress.getAddress(), Is.is(Matchers.nullValue()));
        MatcherAssert.assertThat(unresolvableHostAddress.getHostName(), Is.is("somehost"));
        MatcherAssert.assertThat(unresolvableHostAddress.getHostString(), Is.is("somehost"));
    }

    @Test
    void testCreateDatagramPacket_withoutDestinationAddress_checkProperties() {
        DatagramPacket packet = new DatagramPacket(new byte[0], 0);
        MatcherAssert.assertThat(packet.getPort(), Is.is(-1));
        MatcherAssert.assertThat(packet.getAddress(), Is.is(Matchers.nullValue()));
        Assertions.assertThrows(IllegalArgumentException.class, packet::getSocketAddress);
        packet.setPort(33);
        MatcherAssert.assertThat(packet.getPort(), Is.is(33));
        MatcherAssert.assertThat(packet.getAddress(), Is.is(Matchers.nullValue()));
        Assertions.assertDoesNotThrow(packet::getSocketAddress);
        MatcherAssert.assertThat(packet.getSocketAddress().toString(), Is.is("0.0.0.0/0.0.0.0:33"));
    }

    @Test
    void testCreateDatagramPacket_withDestinationAddressExplicitlySetToUnconfigured_checkProperties() {
        DatagramPacket packet = new DatagramPacket(new byte[0], 0, InetAddress.getLoopbackAddress(), 33127);
        MatcherAssert.assertThat(packet.getSocketAddress().toString(), Is.is("localhost/127.0.0.1:33127"));
        packet.setAddress(null);
        MatcherAssert.assertThat(packet.getAddress(), Is.is(Matchers.nullValue()));
        packet.setPort(0);
        MatcherAssert.assertThat(packet.getPort(), Is.is(0));
    }

    @Test
    void testIsSocketAvailable_reportRepeaterPortStatus() throws UnknownHostException {
        InetSocketAddress wildcardAddress = new InetSocketAddress(InetAddress.getLocalHost(), 5065);
        boolean portAvailable = UdpSocketUtilities.isSocketAvailable(wildcardAddress);
        logger.info("The repeater port availability is: " + portAvailable);
    }

    @Test
    void testIsSocketAvailable_noSocketsCreated_returnsTrue() {
        InetSocketAddress wildcardAddress = new InetSocketAddress(11111);
        MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardAddress), Is.is(true));
    }

    @RepeatedTest(value=100)
    void testIsSocketAvailable_MultipleOpenAndClose() throws SocketException {
        InetSocketAddress wildcardAddress = new InetSocketAddress(11111);
        MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardAddress), Is.is(true));
        try (DatagramSocket ignored = UdpSocketUtilities.createBroadcastAwareListeningSocket(11111, true);){
            MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardAddress), Is.is(false));
        }
    }

    @Test
    void testIsSocketAvailable_ThreadSafety() {
        int testPort = 11111;
        int numThreads = 100;
        ExecutorService executorService = Executors.newFixedThreadPool(100);
        InetSocketAddress wildcardAddress = new InetSocketAddress(11111);
        for (int i2 = 0; i2 < 100; ++i2) {
            executorService.submit(() -> {
                MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardAddress), Is.is(true));
                try (DatagramSocket ignored = UdpSocketUtilities.createBroadcastAwareListeningSocket(11111, true);){
                    MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardAddress), Is.is(false));
                }
                catch (SocketException e) {
                    e.printStackTrace();
                }
                logger.info("Thread completed.");
            });
            logger.info("Thread: " + i2 + " submitted");
        }
        executorService.shutdown();
    }

    @CsvSource(value={"0.0.0.0,false", "0.0.0.0,true", "127.0.0.1,false", "127.0.0.1,true"})
    @ParameterizedTest
    void testIsSocketAvailableReturnsFalse_whenSubProcessReservesSocket(String address, boolean socketIsShareable) throws IOException, InterruptedException {
        int testPort = 2222;
        InetSocketAddress wildcardSocketAddress = new InetSocketAddress(2222);
        try (DatagramSocket ignored = UdpSocketUtilities.createBroadcastAwareListeningSocket(2222, socketIsShareable);){
            MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardSocketAddress), Is.is(false));
        }
        MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardSocketAddress), Is.is(true));
        int socketReserveTimeInMilliseconds = 3000;
        JavaProcessManager processManager = UdpSocketReserver.start(address, 2222, 3000);
        Thread.sleep(1500L);
        MatcherAssert.assertThat(processManager.isAlive(), Is.is(true));
        MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardSocketAddress), Is.is(false));
        processManager.waitFor(5000L, TimeUnit.MILLISECONDS);
        MatcherAssert.assertThat(processManager.isAlive(), Is.is(false));
        MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardSocketAddress), Is.is(true));
    }

    @CsvSource(value={"0.0.0.0,false", "0.0.0.0,true", "127.0.0.1,false"})
    @ParameterizedTest
    void testSubProcessFailsToReserveSocket_whenReservedByParentProcess(String address, boolean socketIsShareable) throws IOException, InterruptedException {
        int testPort = 9999;
        InetSocketAddress wildcardSocketAddress = new InetSocketAddress(9999);
        try (DatagramSocket ignored = UdpSocketUtilities.createBroadcastAwareListeningSocket(9999, socketIsShareable);){
            MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardSocketAddress), Is.is(false));
            int socketReserveTimeInMilliseconds = 10000;
            JavaProcessManager processManager = UdpSocketReserver.start(address, 9999, 10000);
            processManager.waitFor(5000L, TimeUnit.MILLISECONDS);
            MatcherAssert.assertThat(processManager.isAlive(), Is.is(false));
            MatcherAssert.assertThat(processManager.getExitValue(), Is.is(Matchers.not(0)));
        }
        MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardSocketAddress), Is.is(true));
    }

    @ValueSource(booleans={false, true})
    @ParameterizedTest
    void testIsSocketAvailable_wildcardAddressShareability(boolean shareable) throws SocketException {
        InetSocketAddress wildcardSocketAddress = new InetSocketAddress(22222);
        MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardSocketAddress), Is.is(true));
        try (DatagramSocket socketInUse = UdpSocketUtilities.createUnboundSendSocket();){
            socketInUse.setReuseAddress(shareable);
            socketInUse.bind(wildcardSocketAddress);
            MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(wildcardSocketAddress), Is.is(false));
        }
    }

    @ValueSource(booleans={false, true})
    @ParameterizedTest
    void testIsSocketAvailable_localHostAddressShareability(boolean shareable) throws SocketException, UnknownHostException {
        InetSocketAddress localHostSocketAddress = new InetSocketAddress(InetAddress.getLocalHost(), 33333);
        MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(localHostSocketAddress), Is.is(true));
        try (DatagramSocket socketInUse = UdpSocketUtilities.createUnboundSendSocket();){
            socketInUse.setReuseAddress(shareable);
            socketInUse.bind(localHostSocketAddress);
            MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(localHostSocketAddress), Is.is(false));
        }
    }

    @ValueSource(booleans={false, true})
    @ParameterizedTest
    void testIsSocketAvailable_loopbackAddressShareability(boolean shareable) throws SocketException {
        InetSocketAddress loopbackSocketAddress = new InetSocketAddress(InetAddress.getLoopbackAddress(), 44444);
        MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(loopbackSocketAddress), Is.is(true));
        try (DatagramSocket socketInUse = UdpSocketUtilities.createUnboundSendSocket();){
            socketInUse.setReuseAddress(shareable);
            socketInUse.bind(loopbackSocketAddress);
            MatcherAssert.assertThat(UdpSocketUtilities.isSocketAvailable(loopbackSocketAddress), Is.is(false));
        }
    }

    @ValueSource(booleans={false, true})
    @ParameterizedTest
    void testCreateEphemeralSendSocketProperties(boolean broadcastEnable) throws SocketException {
        DatagramSocket socketReferenceCopy;
        try (DatagramSocket sendSocket = UdpSocketUtilities.createEphemeralSendSocket(broadcastEnable);){
            socketReferenceCopy = sendSocket;
            MatcherAssert.assertThat(sendSocket.isBound(), Is.is(true));
            MatcherAssert.assertThat(sendSocket.getLocalAddress().toString(), Is.is("0.0.0.0/0.0.0.0"));
            MatcherAssert.assertThat(sendSocket.getLocalPort(), Matchers.not(Is.is(0)));
            MatcherAssert.assertThat(sendSocket.isConnected(), Is.is(false));
            MatcherAssert.assertThat(sendSocket.getPort(), Is.is(-1));
            MatcherAssert.assertThat(sendSocket.getInetAddress(), Is.is(Matchers.nullValue()));
            MatcherAssert.assertThat(sendSocket.getRemoteSocketAddress(), Is.is(Matchers.nullValue()));
            MatcherAssert.assertThat(sendSocket.isClosed(), Is.is(false));
            MatcherAssert.assertThat(sendSocket.getSoTimeout(), Is.is(0));
            MatcherAssert.assertThat(sendSocket.getBroadcast(), Is.is(broadcastEnable));
            MatcherAssert.assertThat(sendSocket.getReuseAddress(), Is.is(false));
            sendSocket.connect(new InetSocketAddress(InetAddress.getLoopbackAddress(), 1234));
            MatcherAssert.assertThat(sendSocket.isConnected(), Is.is(true));
            MatcherAssert.assertThat(sendSocket.isClosed(), Is.is(false));
            MatcherAssert.assertThat(sendSocket.getRemoteSocketAddress().toString(), Is.is("localhost/127.0.0.1:1234"));
        }
        MatcherAssert.assertThat(socketReferenceCopy, Matchers.notNullValue());
        MatcherAssert.assertThat(socketReferenceCopy.isConnected(), Is.is(true));
        MatcherAssert.assertThat(socketReferenceCopy.isClosed(), Is.is(true));
        MatcherAssert.assertThat(socketReferenceCopy.isBound(), Is.is(true));
    }

    @Test
    void testCreateUnboundSendSocketProperties() throws IOException {
        DatagramSocket socketReferenceCopy;
        try (DatagramSocket sendSocket = UdpSocketUtilities.createUnboundSendSocket();){
            socketReferenceCopy = sendSocket;
            MatcherAssert.assertThat(sendSocket.isBound(), Is.is(false));
            MatcherAssert.assertThat(sendSocket.getLocalAddress().toString(), Is.is("0.0.0.0/0.0.0.0"));
            MatcherAssert.assertThat(sendSocket.getLocalPort(), Is.is(0));
            MatcherAssert.assertThat(sendSocket.isConnected(), Is.is(false));
            MatcherAssert.assertThat(sendSocket.getPort(), Is.is(-1));
            MatcherAssert.assertThat(sendSocket.getInetAddress(), Is.is(Matchers.nullValue()));
            MatcherAssert.assertThat(sendSocket.getRemoteSocketAddress(), Is.is(Matchers.nullValue()));
            MatcherAssert.assertThat(sendSocket.isClosed(), Is.is(false));
            MatcherAssert.assertThat(sendSocket.getSoTimeout(), Is.is(0));
            MatcherAssert.assertThat(sendSocket.getBroadcast(), Is.is(true));
            MatcherAssert.assertThat(sendSocket.getReuseAddress(), Is.is(false));
            DatagramPacket datagramPacket = new DatagramPacket(new byte[]{-86, -69}, 2, new InetSocketAddress(InetAddress.getLoopbackAddress(), 9998));
            sendSocket.send(datagramPacket);
            MatcherAssert.assertThat(sendSocket.isBound(), Is.is(true));
            MatcherAssert.assertThat(sendSocket.getLocalAddress().toString(), Is.is("0.0.0.0/0.0.0.0"));
            MatcherAssert.assertThat(sendSocket.getLocalPort(), Matchers.not(Is.is(0)));
            SocketException throwable = Assertions.assertThrows(SocketException.class, () -> sendSocket.bind(new InetSocketAddress(InetAddress.getLocalHost(), 9999)));
            MatcherAssert.assertThat(throwable.getMessage(), Is.is("already bound"));
            sendSocket.connect(new InetSocketAddress(InetAddress.getLoopbackAddress(), 1234));
            MatcherAssert.assertThat(sendSocket.isConnected(), Is.is(true));
            MatcherAssert.assertThat(sendSocket.isClosed(), Is.is(false));
            MatcherAssert.assertThat(sendSocket.getRemoteSocketAddress().toString(), Is.is("localhost/127.0.0.1:1234"));
        }
        MatcherAssert.assertThat(socketReferenceCopy, Matchers.notNullValue());
        MatcherAssert.assertThat(socketReferenceCopy.isConnected(), Is.is(true));
        MatcherAssert.assertThat(socketReferenceCopy.isClosed(), Is.is(true));
        MatcherAssert.assertThat(socketReferenceCopy.isBound(), Is.is(true));
        MatcherAssert.assertThat(socketReferenceCopy.isBound(), Is.is(true));
    }

    @Test
    void testCreateUnboundSendSocket_BindBehaviourFollowingConnect() throws IOException {
        try (DatagramSocket sendSocket = UdpSocketUtilities.createUnboundSendSocket();){
            MatcherAssert.assertThat(sendSocket.isBound(), Is.is(false));
            sendSocket.connect(new InetSocketAddress(InetAddress.getLoopbackAddress(), 1234));
            MatcherAssert.assertThat(sendSocket.isBound(), Is.is(true));
        }
    }

    @ValueSource(strings={"127.0.0.1", "240.0.0.0", "google.com"})
    @ParameterizedTest
    void testSocketSend_investigateMaximumLengthOfDatagram(String inetAddress) throws UnknownHostException {
        int bufferSize = 100000;
        DatagramPacket packet = new DatagramPacket(new byte[100000], 100000, InetAddress.getByName(inetAddress), 7712);
        IOException exception = null;
        int maxDatagramLength = 100000;
        try (DatagramSocket sendSocket = UdpSocketUtilities.createEphemeralSendSocket(true);){
            for (int i2 = 0; i2 < 100000; i2 += 1000) {
                packet.setLength(i2);
                sendSocket.send(packet);
                maxDatagramLength = i2;
            }
        }
        catch (IOException ex) {
            exception = ex;
        }
        Assertions.assertNotNull(exception);
        MatcherAssert.assertThat(exception, Matchers.isA(IOException.class));
        MatcherAssert.assertThat(exception.getMessage(), Matchers.anyOf(Matchers.containsString("The message is larger than the maximum supported by the underlying transport: Datagram send failed"), Matchers.containsString("Network is unreachable: Datagram send failed"), Matchers.containsString("Message too long"), Matchers.containsString("sendto failed")));
        logger.info("The exception message details were: '" + exception.getMessage() + "'.");
        logger.info("The maximum datagram length for InetAddress: '" + inetAddress + "' is " + maxDatagramLength + " bytes.");
    }

    @Test
    void testSocketSend_investigateAttemptToBroadcastWhenSocketNotBroadcastEnabled() throws UnknownHostException, SocketException {
        Exception exception;
        DatagramPacket packet = new DatagramPacket(new byte[]{-86}, 1, InetAddress.getByName("255.255.255.255"), 6904);
        try (DatagramSocket sendSocket = UdpSocketUtilities.createEphemeralSendSocket(false);){
            exception = Assertions.assertThrows(IOException.class, () -> sendSocket.send(packet));
        }
        MatcherAssert.assertThat(exception, Matchers.isA(IOException.class));
        MatcherAssert.assertThat(exception.getMessage(), Matchers.anyOf(Matchers.containsString("Permission denied"), Matchers.containsString("Can't assign requested address (sendto failed)"), Matchers.containsString("No buffer space available (sendto failed)")));
        logger.info("The exception message details were: '" + exception.getMessage() + "'.");
    }

    @Test
    void testSocketSend_investigateNobodyListening() throws UnknownHostException, SocketException {
        Exception exception;
        DatagramPacket packet = new DatagramPacket(new byte[]{-86}, 1, InetAddress.getByName("100::"), 44231);
        try (DatagramSocket sendSocket = UdpSocketUtilities.createEphemeralSendSocket(false);){
            exception = Assertions.assertThrows(IOException.class, () -> sendSocket.send(packet));
        }
        MatcherAssert.assertThat(exception.getMessage(), Matchers.anyOf(Matchers.containsString("Protocol not allowed"), Matchers.containsString("Network is unreachable"), Matchers.containsString("No route to host (sendto failed)"), Matchers.containsString("No buffer space available (sendto failed)"), Matchers.containsString("Protocol family unavailable")));
        logger.info("The exception message details were: '" + exception.getMessage() + "'.");
    }

    @ParameterizedTest
    @ValueSource(booleans={true, false})
    void testCreateBroadcastAwareListeningSocket_DefaultProperties(boolean shareable) throws SocketException {
        try (DatagramSocket socket = UdpSocketUtilities.createBroadcastAwareListeningSocket(1234, shareable);){
            MatcherAssert.assertThat(socket.isClosed(), Is.is(false));
            MatcherAssert.assertThat(socket.getBroadcast(), Is.is(true));
            MatcherAssert.assertThat(socket.isConnected(), Is.is(false));
            String reason = "The socket reuse mode was not as configured. Perhaps it is not available on the current runtime platform.";
            MatcherAssert.assertThat("The socket reuse mode was not as configured. Perhaps it is not available on the current runtime platform.", socket.getReuseAddress(), Is.is(shareable));
            MatcherAssert.assertThat(socket.isBound(), Is.is(true));
            MatcherAssert.assertThat(socket.getLocalSocketAddress().toString(), Is.is("0.0.0.0/0.0.0.0:1234"));
            MatcherAssert.assertThat(socket.getLocalAddress().toString(), Is.is("0.0.0.0/0.0.0.0"));
            MatcherAssert.assertThat(socket.getLocalPort(), Is.is(1234));
            MatcherAssert.assertThat(socket.getReceiveBufferSize(), Is.is(Matchers.greaterThan(16384)));
            MatcherAssert.assertThat(socket.getRemoteSocketAddress(), Is.is(Matchers.nullValue()));
            MatcherAssert.assertThat(socket.getPort(), Is.is(-1));
            MatcherAssert.assertThat(socket.getInetAddress(), Is.is(Matchers.nullValue()));
            socket.close();
            MatcherAssert.assertThat(socket.isClosed(), Is.is(true));
            MatcherAssert.assertThat(socket.isBound(), Is.is(true));
        }
    }

    @Test
    void testCreateBroadcastAwareListeningSocket_verifyShareablePortsAreShareable() throws SocketException {
        try (DatagramSocket ignored = UdpSocketUtilities.createBroadcastAwareListeningSocket(1234, true);){
            Assertions.assertDoesNotThrow(() -> UdpSocketUtilities.createBroadcastAwareListeningSocket(1234, true).close());
        }
    }

    @Test
    void testCreateBroadcastAwareListeningSocket_verifyUnshareablePortsAreUnshareable() throws SocketException {
        try (DatagramSocket ignored = UdpSocketUtilities.createBroadcastAwareListeningSocket(1234, false);){
            Assertions.assertThrows(SocketException.class, () -> UdpSocketUtilities.createBroadcastAwareListeningSocket(1234, false));
        }
    }

    @Test
    void testCreateBroadcastAwareListeningSocket_autoclose() throws SocketException {
        DatagramSocket socketReferenceCopy;
        try (DatagramSocket listeningSocket = UdpSocketUtilities.createBroadcastAwareListeningSocket(1234, false);){
            socketReferenceCopy = listeningSocket;
            MatcherAssert.assertThat(listeningSocket.isClosed(), Is.is(false));
        }
        MatcherAssert.assertThat(socketReferenceCopy.isClosed(), Is.is(true));
    }

    private static List<Inet4Address> getArgumentsForIntegrationTestBroadcastCapability() throws UnknownHostException {
        InetAddress blacklistAddress = Inet4Address.getByName("192.168.251.255");
        List<Inet4Address> allAddresses = NetworkUtilities.getLocalBroadcastAddresses();
        return allAddresses.stream().filter(x -> !x.equals(blacklistAddress)).collect(Collectors.toList());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @MethodSource(value={"getArgumentsForIntegrationTestBroadcastCapability"})
    @ParameterizedTest
    void integrationTestBroadcastCapability(Inet4Address broadcastAddress) throws IOException, ExecutionException, InterruptedException {
        logger.info("Testing broadcast address: '" + broadcastAddress + "'.");
        if (NetworkUtilities.isVpnActive()) {
            logger.warning("This test is not supported when a VPN connection is active on the local network interface.");
            return;
        }
        int testPort = 8888;
        DatagramPacket receivePacket = new DatagramPacket(new byte[10], 10);
        try (DatagramSocket listenSocket = UdpSocketUtilities.createBroadcastAwareListeningSocket(8888, false);
             DatagramSocket sendSocket = UdpSocketUtilities.createUnboundSendSocket();){
            ExecutorService executor = Executors.newSingleThreadExecutor();
            Future<?> f = executor.submit(() -> {
                logger.info("Receive thread starting to listen on address: '" + listenSocket.getLocalSocketAddress() + "'.");
                try {
                    listenSocket.receive(receivePacket);
                    logger.info("Received new packet from: '" + receivePacket.getSocketAddress() + "'.");
                }
                catch (Exception ex) {
                    logger.info("Received thread interrupted by exception: " + ex.getMessage() + ".");
                }
                logger.info("Receiver task completed.");
            });
            logger.info("Sending packet...");
            DatagramPacket sendPacket = new DatagramPacket(new byte[]{-86, -69}, 2);
            sendPacket.setSocketAddress(new InetSocketAddress(broadcastAddress, 8888));
            Assertions.assertDoesNotThrow(() -> sendSocket.send(sendPacket), "The send operation generated an exception. Is this test being run behind a VPN where broadcasts are not supported ?");
            logger.info("Send completed.");
            try {
                logger.info("Waiting 500ms for data...");
                f.get(500L, TimeUnit.MILLISECONDS);
                logger.info("Wait terminated.");
                MatcherAssert.assertThat(receivePacket.getLength(), Is.is(sendPacket.getLength()));
                MatcherAssert.assertThat(receivePacket.getData()[0], Is.is(sendPacket.getData()[0]));
                MatcherAssert.assertThat(receivePacket.getData()[1], Is.is(sendPacket.getData()[1]));
            }
            catch (TimeoutException ex) {
                Assertions.fail("Timeout - data not received.");
                f.cancel(true);
            }
            finally {
                logger.info("cleaning up.");
                executor.shutdown();
            }
        }
    }

    @ValueSource(booleans={true, false})
    @ParameterizedTest
    void integrationTestSynchronousDataTransfer(boolean bindSendSocket) throws Exception {
        DatagramPacket receivePacket = new DatagramPacket(new byte[10], 10);
        DatagramPacket sendPacket = new DatagramPacket(new byte[]{-86, -69}, 2);
        try (DatagramSocket listenSocket = UdpSocketUtilities.createBroadcastAwareListeningSocket(12345, false);
             DatagramSocket sendSocket = bindSendSocket ? UdpSocketUtilities.createEphemeralSendSocket(false) : UdpSocketUtilities.createUnboundSendSocket();){
            MatcherAssert.assertThat(sendSocket.isBound(), Is.is(bindSendSocket));
            sendSocket.connect(InetAddress.getLoopbackAddress(), 12345);
            MatcherAssert.assertThat(sendSocket.isConnected(), Is.is(true));
            logger.info("About to send...");
            sendSocket.send(sendPacket);
            logger.info("Sent.");
            logger.info("Waiting for receive to complete");
            try {
                logger.info("Calling receive()...");
                listenSocket.receive(receivePacket);
                logger.info("Received new packet from: " + receivePacket.getSocketAddress());
            }
            catch (Exception ex) {
                logger.info("Exception thrown !" + ex.getMessage());
            }
            logger.info("Receive thread completed");
        }
        MatcherAssert.assertThat(receivePacket.getLength(), Is.is(2));
        MatcherAssert.assertThat(receivePacket.getSocketAddress() instanceof InetSocketAddress, Is.is(true));
        MatcherAssert.assertThat(((InetSocketAddress)receivePacket.getSocketAddress()).getAddress(), Matchers.not(Is.is(InetAddress.getByName("0.0.0.0"))));
        MatcherAssert.assertThat(((InetSocketAddress)receivePacket.getSocketAddress()).getPort(), Matchers.not(Is.is(0)));
        MatcherAssert.assertThat(receivePacket.getData()[0], Is.is(sendPacket.getData()[0]));
        MatcherAssert.assertThat(receivePacket.getData()[1], Is.is(sendPacket.getData()[1]));
    }

    @ValueSource(booleans={true, false})
    @ParameterizedTest
    void integrationTestAsynchronousDataTransfer(boolean bindSendSocket) throws Exception {
        DatagramPacket receivePacket = new DatagramPacket(new byte[10], 10);
        DatagramPacket sendPacket = new DatagramPacket(new byte[]{-86, -69}, 2);
        try (DatagramSocket listenSocket = UdpSocketUtilities.createBroadcastAwareListeningSocket(12345, false);
             DatagramSocket sendSocket = bindSendSocket ? UdpSocketUtilities.createEphemeralSendSocket(false) : UdpSocketUtilities.createUnboundSendSocket();){
            MatcherAssert.assertThat(sendSocket.isBound(), Is.is(bindSendSocket));
            sendSocket.connect(InetAddress.getLoopbackAddress(), 12345);
            MatcherAssert.assertThat(sendSocket.isConnected(), Is.is(true));
            ExecutorService executor = Executors.newSingleThreadExecutor();
            Future<?> f = executor.submit(() -> {
                logger.info("Receive thread started");
                try {
                    logger.info("Calling receive()...");
                    listenSocket.receive(receivePacket);
                    logger.info("Received new packet from: " + receivePacket.getSocketAddress());
                }
                catch (Exception ex) {
                    logger.info("Exception thrown !" + ex.getMessage());
                }
                logger.info("Receive thread completed");
            });
            logger.info("About to send...");
            sendSocket.send(sendPacket);
            logger.info("Sent. Waiting for receive to complete");
            f.get();
            logger.info("Receive completed");
            executor.shutdown();
        }
        MatcherAssert.assertThat(receivePacket.getLength(), Is.is(2));
        MatcherAssert.assertThat(receivePacket.getSocketAddress() instanceof InetSocketAddress, Is.is(true));
        MatcherAssert.assertThat(((InetSocketAddress)receivePacket.getSocketAddress()).getAddress(), Matchers.not(Is.is(InetAddress.getByName("0.0.0.0"))));
        MatcherAssert.assertThat(((InetSocketAddress)receivePacket.getSocketAddress()).getPort(), Matchers.not(Is.is(0)));
        MatcherAssert.assertThat(receivePacket.getData()[0], Is.is(sendPacket.getData()[0]));
        MatcherAssert.assertThat(receivePacket.getData()[1], Is.is(sendPacket.getData()[1]));
    }
}

