/*
 * Decompiled with CFR 0.152.
 */
package org.epics.ca.impl.repeater;

import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.jcip.annotations.ThreadSafe;
import org.apache.commons.lang3.Validate;
import org.epics.ca.impl.ProtocolConfiguration;
import org.epics.ca.impl.repeater.CARepeaterClientManager;
import org.epics.ca.impl.repeater.CARepeaterClientProxy;
import org.epics.ca.impl.repeater.CARepeaterMessage;
import org.epics.ca.impl.repeater.CARepeaterStatusChecker;
import org.epics.ca.impl.repeater.NetworkUtilities;
import org.epics.ca.impl.repeater.UdpSocketUtilities;
import org.epics.ca.util.logging.LibraryLogManager;
import org.epics.ca.util.net.InetAddressUtil;

@ThreadSafe
class CARepeater {
    private static final Logger logger = LibraryLogManager.getLogger(CARepeater.class);
    private final byte[] buffer;
    private final ByteBuffer data;
    private final ExecutorService executorService;
    private final AtomicBoolean shutdownRequest;
    private final CARepeaterClientManager clientProxyManager;
    private final DatagramSocket listeningSocket;

    public static void main(String[] argv) {
        CARepeater repeater;
        int port;
        if (!NetworkUtilities.verifyTargetPlatformNetworkStackIsChannelAccessCompatible()) {
            System.exit(128);
        }
        logger.info("The CA Repeater main method has been invoked with " + argv.length + " arguments.");
        logger.info("The arguments were: " + Arrays.toString(argv));
        boolean portArgumentSupplied = argv.length >= 2 && (argv[0].equals("-p") || argv[0].equals("--port"));
        int fallbackRepeaterPort = new ProtocolConfiguration().getRepeaterPort();
        int n = port = portArgumentSupplied ? CARepeater.parseToInt(argv[1], fallbackRepeaterPort) : fallbackRepeaterPort;
        if (CARepeaterStatusChecker.isRepeaterRunning(port)) {
            logger.info("The repeater is already running and a new instance will not be started.");
            logger.info("This process will now terminate.");
            System.exit(129);
        }
        try {
            logger.info("Creating CA Repeater instance which will run on port " + port + ".");
            repeater = new CARepeater(port);
        }
        catch (CaRepeaterStartupException ex) {
            logger.warning("An exception occurred when attempting to start the repeater.");
            logger.warning("The exception message was: " + ex.getMessage());
            System.exit(130);
            return;
        }
        repeater.start();
    }

    CARepeater(int repeaterPort) throws CaRepeaterStartupException {
        Validate.inclusiveBetween(1L, 65535L, repeaterPort, "The port must be in the range 1-65535.");
        logger.info("Creating CA repeater instance which will bind to the wildcard address on port " + repeaterPort + ".");
        this.buffer = new byte[65551];
        this.data = ByteBuffer.wrap(this.buffer);
        this.shutdownRequest = new AtomicBoolean(false);
        this.executorService = Executors.newSingleThreadExecutor();
        try {
            logger.finest("Creating broadcast-aware listening socket on port: " + repeaterPort);
            this.listeningSocket = UdpSocketUtilities.createBroadcastAwareListeningSocket(repeaterPort, false);
            logger.finest("The listening socket was created ok.");
        }
        catch (SocketException ex) {
            String msg = "An unexpected exception has prevented the CA Repeater from starting.";
            logger.log(Level.WARNING, "An unexpected exception has prevented the CA Repeater from starting.", ex);
            throw new CaRepeaterStartupException("An unexpected exception has prevented the CA Repeater from starting.", ex);
        }
        InetSocketAddress repeaterListeningSocketAddress = (InetSocketAddress)this.listeningSocket.getLocalSocketAddress();
        logger.finest("The repeater will advertise its availability on the socket with address : '" + repeaterListeningSocketAddress);
        this.clientProxyManager = new CARepeaterClientManager(repeaterListeningSocketAddress);
    }

    void start() {
        if (this.executorService.isShutdown()) {
            throw new IllegalStateException("This CA Repeater instance has been shutdown and cannot be restarted.");
        }
        Future<?> task = this.executorService.submit(() -> {
            try {
                this.processUdpDatagramPackets();
            }
            catch (Exception ex) {
                logger.info("An unrecoverable exception has occurred which means this CA repeater will terminate.");
            }
            logger.info("The CA Repeater has terminated.");
            this.executorService.shutdown();
        });
    }

    void shutdown() {
        if (this.executorService.isShutdown()) {
            throw new IllegalStateException("This CA Repeater instance has already been shutdown.");
        }
        this.shutdownRequest.set(true);
        this.listeningSocket.close();
    }

    private void processUdpDatagramPackets() {
        logger.finest("Processing incoming UDP datagrams...");
        while (!this.shutdownRequest.get()) {
            try {
                logger.finest("Waiting for next datagram.");
                DatagramPacket inputPacket = this.waitForDatagram();
                if (this.shutdownRequest.get()) {
                    logger.finest("The wait for the next datagram has terminated.");
                    logger.finest("The CA repeater has been shutdown. Will not process any more messages.");
                    return;
                }
                logger.finest("A new UDP datagram packet has been received. ");
                boolean unprocessedMessages = true;
                DatagramPacket packetToProcess = inputPacket;
                while (unprocessedMessages) {
                    logger.finest("Consuming next message in UDP datagram packet.");
                    logger.finest("The length of the UDP datagram is: " + packetToProcess.getLength() + " bytes.");
                    DatagramPacket residualMessagePacket = this.processOneMessage(packetToProcess, this::handleClientRegistrationRequest, this::handleClientRegistrationRequest, this::handleBeaconMessage, this::handleAllOtherMessages);
                    logger.finest("After processing the length of the UDP datagram is: " + residualMessagePacket.getLength() + " bytes.");
                    unprocessedMessages = residualMessagePacket.getLength() > 0;
                    packetToProcess = residualMessagePacket;
                }
            }
            catch (Exception ex) {
                logger.log(Level.WARNING, "An exception was thrown whilst waiting for, or processing, a datagram.", ex);
            }
        }
    }

    private DatagramPacket processOneMessage(DatagramPacket inputPacket, Consumer<DatagramPacket> zeroLengthMessageHandler, Consumer<DatagramPacket> clientRegistrationMessageHandler, Consumer<DatagramPacket> beaconMessageHandler, Consumer<DatagramPacket> defaultMessageHandler) {
        Validate.notNull(zeroLengthMessageHandler);
        Validate.notNull(clientRegistrationMessageHandler);
        Validate.notNull(zeroLengthMessageHandler);
        Validate.notNull(defaultMessageHandler);
        logger.finest("Consuming one message.");
        int bytesReceived = inputPacket.getLength();
        logger.finest("The length of the UDP datagram packet is " + bytesReceived + " bytes.");
        InetSocketAddress senderSocketAddress = (InetSocketAddress)inputPacket.getSocketAddress();
        logger.finest("The message was sent from socket '" + senderSocketAddress + "'");
        if (bytesReceived == 0) {
            logger.finest("Calling ZERO LENGTH MESSAGE consumer.");
            zeroLengthMessageHandler.accept(inputPacket);
            return inputPacket;
        }
        if (bytesReceived >= 16) {
            ByteBuffer buffer = ByteBuffer.wrap(inputPacket.getData());
            short commandCode = buffer.getShort(CARepeaterMessage.CaHeaderOffsets.CA_HDR_SHORT_COMMAND_OFFSET.value);
            if (commandCode == CARepeaterMessage.CaCommandCodes.CA_REPEATER_REGISTER.value) {
                logger.finest("Calling CLIENT REGISTRATION MESSAGE consumer.");
                clientRegistrationMessageHandler.accept(inputPacket);
                logger.finest("Removing processed CLIENT REGISTRATION MESSAGE.");
                return CARepeater.removeProcessedMessage(inputPacket, 16);
            }
            if (commandCode == CARepeaterMessage.CaCommandCodes.CA_PROTO_RSRV_IS_UP.value) {
                logger.finest("Calling BEACON MESSAGE consumer.");
                beaconMessageHandler.accept(inputPacket);
                logger.finest("Removing processed BEACON MESSAGE.");
                return CARepeater.removeProcessedMessage(inputPacket, 16);
            }
        }
        logger.finest("Calling DEFAULT MESSAGE consumer.");
        defaultMessageHandler.accept(inputPacket);
        logger.finest("Removing processed DEFAULT MESSAGE of length " + inputPacket.getLength() + " bytes.");
        return CARepeater.removeProcessedMessage(inputPacket, inputPacket.getLength());
    }

    private void handleClientRegistrationRequest(DatagramPacket packet) {
        Validate.notNull(packet);
        Validate.isTrue(packet.getLength() == 0 || packet.getLength() >= 16);
        logger.finest("Handling CA Client Registration Message sent from socket '" + packet.getSocketAddress() + "'");
        ByteBuffer buffer = ByteBuffer.wrap(packet.getData());
        InetAddress serverInetAddress = InetAddressUtil.intToIPv4Address(buffer.getInt(CARepeaterMessage.CaHeaderOffsets.CA_HDR_INT_PARAM2_OFFSET.value));
        InetSocketAddress serverSocketAddress = new InetSocketAddress(serverInetAddress, packet.getPort());
        logger.finest("The server address encoded in the datagram was: '" + serverInetAddress + "'");
        if (!NetworkUtilities.isThisMyIpAddress(packet.getAddress())) {
            logger.warning("The internet address associated with the request datagram (" + packet.getAddress() + ") was not a local address.'");
            logger.warning("The CA repeater can only register clients on one of the local machine interfaces.");
            return;
        }
        if (CARepeaterClientProxy.isClientDead((InetSocketAddress)packet.getSocketAddress())) {
            logger.warning("The CA repeater client (" + packet.getAddress() + ") reports that it is dead.");
            logger.warning("The CA repeater only register clients that are alive.");
            return;
        }
        if (this.clientProxyManager.isListeningPortAlreadyAssigned(packet.getPort())) {
            logger.warning("The internet address associated with the request datagram (" + packet.getSocketAddress() + ") is already a registered client.");
            logger.warning("Nothing further to do.");
            return;
        }
        InetSocketAddress clientListeningSocket = new InetSocketAddress(packet.getAddress(), packet.getPort());
        this.clientProxyManager.registerNewClient(clientListeningSocket);
    }

    private void handleBeaconMessage(DatagramPacket packet) {
        Validate.notNull(packet);
        Validate.isTrue(packet.getLength() >= 16);
        logger.finest("Handling CA Beacon Message sent from socket '" + packet.getSocketAddress() + "'.");
        ByteBuffer buffer = ByteBuffer.wrap(packet.getData());
        short caServerVersionNumber = buffer.getShort(CARepeaterMessage.CaBeaconMessageOffsets.CA_HDR_SHORT_BEACON_MSG_SERVER_PROTOCOL_MINOR_VERSION_OFFSET.value);
        logger.finest("The CA Beacon Message indicates the server's protocol minor version number is: '" + caServerVersionNumber + "'.");
        short serverListeningPort = buffer.getShort(CARepeaterMessage.CaBeaconMessageOffsets.CA_HDR_SHORT_BEACON_MSG_SERVER_TCP_LISTENING_PORT_OFFSET.value);
        logger.finest("The CA Beacon Message indicates the server's TCP listening port is: '" + serverListeningPort + "'");
        int serverBeaconId = buffer.getInt(CARepeaterMessage.CaBeaconMessageOffsets.CA_HDR_INT_BEACON_MSG_SERVER_BEACON_ID_OFFSET.value);
        logger.finest("The CA Beacon Message has the following Beacon ID '" + serverBeaconId + "'.");
        InetSocketAddress excludeForwardingToSelfSocketAddress = (InetSocketAddress)packet.getSocketAddress();
        int serverAddressEncodedInMessage = buffer.getInt(CARepeaterMessage.CaBeaconMessageOffsets.CA_HDR_INT_BEACON_MSG_SERVER_ADDR_OFFSET.value);
        String serverAddressEncodedInMessageAsString = InetAddressUtil.intToIPv4Address(serverAddressEncodedInMessage).toString();
        logger.finest("The CA Beacon Message advertised the server's IP address as being: '" + serverAddressEncodedInMessageAsString + "'.");
        if (serverAddressEncodedInMessage == 0) {
            logger.finest("Using IP address from datagram sending socket (" + packet.getAddress() + ").");
            logger.finest("Forwarding Beacon Message...");
            InetAddress serverAddressEncodedInDatagram = packet.getAddress();
            this.clientProxyManager.forwardBeacon(caServerVersionNumber, serverListeningPort, serverBeaconId, serverAddressEncodedInDatagram, excludeForwardingToSelfSocketAddress);
        } else {
            logger.finest("Using IP address encoded in message (" + serverAddressEncodedInMessageAsString + ").");
            logger.finest("Forwarding Beacon Message...");
            this.clientProxyManager.forwardBeacon(caServerVersionNumber, serverListeningPort, serverBeaconId, InetAddressUtil.intToIPv4Address(serverAddressEncodedInMessage), excludeForwardingToSelfSocketAddress);
        }
    }

    private void handleAllOtherMessages(DatagramPacket packet) {
        Validate.notNull(packet);
        Validate.isTrue(packet.getSocketAddress() instanceof InetSocketAddress);
        InetSocketAddress excludeForwardingToSelfSocketAddress = (InetSocketAddress)packet.getSocketAddress();
        this.clientProxyManager.forwardDatagram(packet, excludeForwardingToSelfSocketAddress);
    }

    private DatagramPacket waitForDatagram() throws Exception {
        this.data.clear();
        DatagramPacket packet = new DatagramPacket(this.buffer, this.buffer.length);
        try {
            this.listeningSocket.receive(packet);
        }
        catch (Exception ex) {
            if (this.shutdownRequest.get()) {
                logger.finest("The receive datagram operation terminated because the CA repeater was shutdown.");
                return new DatagramPacket(new byte[0], 0);
            }
            String msg = "An unexpected exception has made it impossible to obtain a new datagram.";
            logger.finest("An unexpected exception has made it impossible to obtain a new datagram.");
            Thread.currentThread().interrupt();
            throw new Exception("An unexpected exception has made it impossible to obtain a new datagram.", ex);
        }
        logger.finest("");
        logger.finest("CA Repeater listening socket has received new data. Processing...");
        return packet;
    }

    private static DatagramPacket removeProcessedMessage(DatagramPacket inputPacket, int messageToRemoveLength) {
        Validate.notNull(inputPacket);
        Validate.isTrue(messageToRemoveLength <= inputPacket.getLength());
        logger.finest("Removing message of length " + messageToRemoveLength + " bytes.");
        int newLength = inputPacket.getLength() - messageToRemoveLength;
        byte[] newPayload = Arrays.copyOfRange(inputPacket.getData(), messageToRemoveLength, inputPacket.getLength());
        SocketAddress newSocketAddress = inputPacket.getSocketAddress();
        DatagramPacket outputPacket = new DatagramPacket(newPayload, newLength, newSocketAddress);
        logger.finest("The datagram packet is now of length " + newLength + " bytes.");
        return outputPacket;
    }

    private static int parseToInt(String stringToParse, int defaultValue) {
        int ret;
        try {
            ret = Integer.parseInt(stringToParse);
        }
        catch (NumberFormatException ex) {
            ret = defaultValue;
        }
        return ret;
    }

    public static class CaRepeaterStartupException
    extends Exception {
        public CaRepeaterStartupException(String message, Exception ex) {
            super(message, ex);
        }
    }
}

