/*
 * Decompiled with CFR 0.152.
 */
package org.epics.ca.impl.repeater;

import java.net.DatagramPacket;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.commons.lang3.Validate;
import org.epics.ca.impl.repeater.CARepeaterClientProxy;
import org.epics.ca.util.logging.LibraryLogManager;

class CARepeaterClientManager {
    private static final Logger logger = LibraryLogManager.getLogger(CARepeaterClientManager.class);
    private final Map<InetSocketAddress, CARepeaterClientProxy> clientMap = Collections.synchronizedMap(new HashMap());
    private final InetSocketAddress repeaterListeningSocketAddress;

    public CARepeaterClientManager(InetSocketAddress repeaterListeningSocketAddress) {
        this.repeaterListeningSocketAddress = (InetSocketAddress)Validate.notNull((Object)repeaterListeningSocketAddress, (String)"The 'repeaterListeningAddress' argument was null.", (Object[])new Object[0]);
    }

    void registerNewClient(InetSocketAddress clientListeningSocketAddress) {
        CARepeaterClientProxy proxy;
        Validate.notNull((Object)clientListeningSocketAddress);
        logger.finest("Attempting to register new client listening at socket address: " + clientListeningSocketAddress);
        try {
            logger.finest("Creating new proxy for CA Repeater Client... ");
            proxy = new CARepeaterClientProxy(clientListeningSocketAddress);
            logger.finest("The proxy was created OK. ");
        }
        catch (SocketException ex) {
            logger.log(Level.WARNING, "The proxy could not be created.", ex);
            return;
        }
        logger.finest("Sending repeater registration confirm message to '" + clientListeningSocketAddress + "'.");
        if (!proxy.sendCaRepeaterConfirmMessage(this.repeaterListeningSocketAddress.getAddress())) {
            proxy.close();
            logger.warning("Failed to send repeater registration confirm message.");
            return;
        }
        logger.finest("Adding new client to list of registered CA repeater clients.");
        this.clientMap.put(clientListeningSocketAddress, proxy);
        logger.finest("The list now contains " + this.clientMap.size() + " clients.");
        logger.finest("Sending repeater protocol version message to registered clients...");
        logger.finest("Will exclude the newly created client..." + clientListeningSocketAddress);
        List<CARepeaterClientProxy> failedNotifications = this.sendVersionMessageToRegisteredClients(clientListeningSocketAddress);
        if (failedNotifications.size() > 0) {
            logger.warning("Failed to send protocol version message to one or more registered clients.");
        }
        logger.finest("Performing housekeeping on registered clients...");
        this.removeDeadClients();
    }

    void forwardBeacon(short serverProtocolMinorVersion, short serverListeningPort, int serverBeaconId, InetAddress serverAddress, InetSocketAddress excluded) {
        Validate.notNull((Object)serverAddress);
        Validate.notNull((Object)excluded);
        logger.finest("Forwarding beacon with ID: " + serverBeaconId + " to " + this.clientMap.size() + " CA Repeater clients.");
        logger.finest("Any CA Repeater client with socket address to " + excluded + " will be excluded from notification.");
        List<CARepeaterClientProxy> failedNotifications = this.sendBeaconMessageToRegisteredClients(serverProtocolMinorVersion, serverListeningPort, serverBeaconId, serverAddress, excluded);
        if (failedNotifications.size() > 0) {
            logger.warning("Failed to send beacon message to one or more registered clients.");
        }
        logger.finest("Performing housekeeping on registered clients...");
        this.removeDeadClients();
    }

    void forwardDatagram(DatagramPacket packet, InetSocketAddress excluded) {
        Validate.notNull((Object)packet);
        Validate.notNull((Object)excluded);
        logger.finest("Forwarding datagram packet to " + this.clientMap.size() + " CA Repeater clients.");
        logger.finest("Any CA Repeater client with socket address to " + excluded + " will be excluded from notification.");
        DatagramPacket sendPacket = new DatagramPacket(packet.getData(), packet.getLength());
        List<CARepeaterClientProxy> failedNotifications = this.sendDatagramToRegisteredClients(sendPacket, excluded);
        logger.finest("There were " + failedNotifications.size() + " send failures.");
        logger.finest("Performing housekeeping on registered clients...");
        this.removeDeadClients();
    }

    boolean isListeningPortAlreadyAssigned(int listeningPort) {
        return this.clientMap.keySet().stream().anyMatch(p -> p.getPort() == listeningPort);
    }

    private List<CARepeaterClientProxy> sendBeaconMessageToRegisteredClients(short serverCaVersionNumber, short serverTcpListeningPort, int serverBeaconId, InetAddress serverAddress, InetSocketAddress excluded) {
        Validate.notNull((Object)serverAddress);
        Validate.notNull((Object)excluded);
        ArrayList<CARepeaterClientProxy> clientList = new ArrayList<CARepeaterClientProxy>(this.clientMap.values());
        logger.finest("Attempting to send 'CA_RSRV_IS_UP' (Beacon) message to " + clientList.size() + " CA Repeater clients.");
        List<CARepeaterClientProxy> failedNotificationList = clientList.stream().filter(client -> !client.getClientListeningSocketAddress().equals(excluded)).filter(client -> !client.sendCaServerBeaconMessage(serverCaVersionNumber, serverTcpListeningPort, serverBeaconId, serverAddress)).collect(Collectors.toList());
        logger.finest("There were " + failedNotificationList.size() + " send failures.");
        return failedNotificationList;
    }

    private List<CARepeaterClientProxy> sendDatagramToRegisteredClients(DatagramPacket packet, InetSocketAddress excluded) {
        Validate.notNull((Object)packet);
        Validate.notNull((Object)excluded);
        Validate.isTrue((packet.getAddress() == null ? 1 : 0) != 0);
        Validate.isTrue((packet.getPort() == -1 ? 1 : 0) != 0);
        ArrayList<CARepeaterClientProxy> clientList = new ArrayList<CARepeaterClientProxy>(this.clientMap.values());
        logger.finest("Attempting to send DATAGRAM to " + clientList.size() + " CA Repeater clients.");
        List<CARepeaterClientProxy> failedNotificationList = clientList.stream().filter(client -> !client.getClientListeningSocketAddress().equals(excluded)).filter(client -> !client.sendDatagram(packet)).collect(Collectors.toList());
        logger.finest("There were " + failedNotificationList.size() + " send failures.");
        return failedNotificationList;
    }

    private List<CARepeaterClientProxy> sendVersionMessageToRegisteredClients(InetSocketAddress excluded) {
        Validate.notNull((Object)excluded);
        ArrayList<CARepeaterClientProxy> clientList = new ArrayList<CARepeaterClientProxy>(this.clientMap.values());
        logger.finest("Attempting to send 'CA_PROTO_VERSION' message to CA Repeater client list containing " + clientList.size() + " CA Repeater clients.");
        logger.finest("The client " + excluded + " will be excluded from CA_PROTO_VERSION notifications.");
        List<CARepeaterClientProxy> failedNotificationList = clientList.stream().filter(client -> !client.getClientListeningSocketAddress().equals(excluded)).filter(client -> !client.sendCaVersionMessage()).collect(Collectors.toList());
        logger.finest("There were " + failedNotificationList.size() + " send failures.");
        return failedNotificationList;
    }

    private void removeDeadClients() {
        logger.finest("Removing any CA Repeater Clients which previously registered but which now are dead.");
        logger.finest("There are currently " + this.clientMap.size() + " clients.");
        ArrayList<CARepeaterClientProxy> clientList = new ArrayList<CARepeaterClientProxy>(this.clientMap.values());
        clientList.stream().filter(CARepeaterClientProxy::isClientDead).forEach(this::removeClient);
        logger.finest("Following dead client cleanup there are now " + this.clientMap.size() + " clients.");
    }

    private void removeClient(CARepeaterClientProxy proxy) {
        logger.finest("Deregistering dead client which used to listen at socket address: " + proxy.getClientListeningSocketAddress());
        CARepeaterClientProxy noLongerRequiredProxy = this.clientMap.remove(proxy.getClientListeningSocketAddress());
        logger.finest("Closing dead client communication proxy.");
        noLongerRequiredProxy.close();
    }
}

