package org.eroq.pathmatcher;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class PathMatcher {
	
	private Pattern pattern;
	private List<PathKey> keys = new ArrayList<>();
	
	public PathMatcher(String path, boolean sensitive, boolean strict, boolean end) {
		List<Object> tokens = parse(path);
		pattern = tokensToRegExp(tokens, sensitive, strict, end);
		for (Object token : tokens) {
			if (!(token instanceof String)) {
				keys.add((PathKey) token);
			}
		}
	}

	public PathMatcher(Pattern pattern) {
		this.pattern = pattern;
		// Obtain count of capturing groups by looking for '(' not followed by '?'
		Pattern p = Pattern.compile("\\((?!\\?)");
		Matcher m = p.matcher(pattern.pattern());
		Integer key=0;
		while(m.find()) {
			keys.add(new PathKey(
				key++,
	    		null,
				null,
				false,
				false,
				null
    	    ));
		}
	}
	
	public Pattern getPattern() {
		return pattern;
	}

	public List<PathKey> getKeys() {
		return keys;
	}
	
	private Pattern tokensToRegExp(List<Object> tokens, boolean sensitive, boolean strict, boolean end) {
		String route = "";
		Object lastToken = tokens.get(tokens.size()-1);
		boolean endsWithSlash = lastToken instanceof String && Pattern.matches("\\/$", (String)lastToken);
		for(Object token: tokens) {
			if(token instanceof String) {
				String stringToken = (String)token;
				route += escapeString(stringToken);
			} else {
				PathKey keyToken = (PathKey)token;
				String prefix = escapeString(keyToken.getPrefix());
				String capture = keyToken.getPattern();
				
				if(keyToken.isRepeat()) {
					capture += "(?:" + prefix + capture + ")*";
				}
				
				if(keyToken.isOptional()) {
					if(!prefix.isEmpty()) {
						capture = "(?:" + prefix + "(" + capture + "))?";
					} else {
						capture = "(" + capture + ")?";
					}
				} else {
					capture = prefix + "(" + capture + ")";
				}
				
				route += capture;
			}
		}
		
		if(!strict) {
			route = (endsWithSlash ? route.substring(0, route.length()-1) : route) + "(?:\\/(?=$))?";
		}
		
		if(end) {
			route += "$";
		} else {
			route += strict && endsWithSlash ? "" : "(?=\\/|$)";
		}
		return Pattern.compile("^" + route, sensitive ? 0 : Pattern.CASE_INSENSITIVE);
	}

	private static final Pattern PATH_REGEXP = Pattern.compile("(\\\\.)|([\\/.])?(?:(?:\\:(\\w+)(?:\\(((?:\\\\.|[^()])+)\\))?|\\(((?:\\\\.|[^()])+)\\))([+*?])?|(\\*))");

	private List<Object> parse(String string) {
		
		List<Object> tokens = new ArrayList<>();
		int key = 0;
		int index = 0;
		String path = "";
				  
		Matcher matcher = PATH_REGEXP.matcher(string);
		while(matcher.find()) {
			String escaped = matcher.group(1);
			int offset = matcher.start();
			path += string.substring(index, offset);
			index = matcher.end();
			
			// Ignore already escaped sequences.
		    if (escaped != null && !escaped.isEmpty()) {
				path += escaped.charAt(1);
				continue;
		    }
		    
		    // Push the current path onto the tokens.
		    if (!path.isEmpty()) {
		      tokens.add(path);
		      path = "";
		    }
		    
		    String prefix = matcher.group(2);
    		String name = matcher.group(3);
    		String capture = matcher.group(4);
			String group = matcher.group(5);
			String suffix = matcher.group(6);
			String asterisk = matcher.group(7);

    	    boolean repeat = suffix!=null && (suffix.equals("+") || suffix.equals("*"));
    		boolean optional = suffix!=null && (suffix.equals("?") || suffix.equals("*"));
    	    String delimiter = prefix != null ? prefix : "/";
    	    String pattern = capture != null ? capture : 
				    	    	( group != null ? group : 
				    	    		(asterisk != null ? ".*" : 
				    	    			"[^" + delimiter + "]+?"));
    	    
    	    tokens.add(new PathKey(
	    		name != null ? name : key++,
	    		prefix != null ? prefix : "",
				delimiter,
				optional,
				repeat,
				escapeGroup(pattern)
    	    ));
    	    
		}
		
	    // Match any characters still remaining.
	    if (index < string.length()) {
	      path += string.substring(index);
	    }

	    // If the path exists, push it onto the end.
	    if (!path.isEmpty()) {
	      tokens.add(path);
	    }

	    return tokens;
	}

	private String escapeString(String string) {
		return string.replace("([.+*?=^!:${}()[\\]|\\/])", "\\$1");
	}

	private String escapeGroup(String group) {
		return group.replace("([=!:$\\/()])", "\\$1");
	}
	
	@Override
	public String toString() {
		StringBuilder repr = new StringBuilder();
		repr.append("Pattern: ");
		repr.append(pattern);
		repr.append(" Keys: [");
		for(PathKey key: keys) {
			repr.append(key);
		}
		repr.append("]");
		return repr.toString();
	}

	@Override
	public boolean equals(Object obj) {
		if(this == obj) {
			return true;
		}
		if(!(obj instanceof PathMatcher)) {
			return false;
		}
		PathMatcher pathMatcherObj = (PathMatcher) obj;
		boolean result = true;
		result &= pattern.pattern().equals(pathMatcherObj.pattern.pattern());
		result &= keys.size() == pathMatcherObj.keys.size();
		if(!result) {
			return false;
		}				
		for(int i=0; i<keys.size(); i++) {
			result &= keys.get(i).equals(pathMatcherObj.keys.get(i));
			if(!result) {
				break;
			}
		}
		return result;
	}
}
