package org.eroq.router;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

import org.eroq.http.Http;
import org.eroq.http.HttpStatus;
import org.eroq.http.Http.HttpServer;
import org.eroq.http.Http.HttpServerListenCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class App extends Router {
	
	private static Logger logger = LoggerFactory.getLogger(App.class);
	
	public App() {
		init();
		loadDefaultProperties();
		initProperties("app.properties");
		printCurrentProperties();
	}
	
	public App(String propertiesFile) {
		init();
		loadDefaultProperties();
		initProperties(propertiesFile);
		printCurrentProperties();
	}
	
	public App(URL propertiesFileURL) {
		init();
		loadDefaultProperties();
		initProperties(propertiesFileURL);
		printCurrentProperties();
	}

	public App(InputStream propertiesInputStream) {
		init();
		loadDefaultProperties();
		initProperties(propertiesInputStream);
		printCurrentProperties();
	}
	
	private long startTime;
	
	private void init() {
		startTime = System.nanoTime();
		logger.info("App starting");		
	}

	private void done(HttpServer server) {
		int durationInMicroSeconds = Math.round((System.nanoTime() - startTime) / 1000.0f);
		logger.info("App started in {}s at http://{}:{}", String.format("%.3f", durationInMicroSeconds/1000000.0f), server.getHost(), server.getPort());		
	}

	private void loadDefaultProperties() {
		logger.debug("Set default properties");
		properties = new Properties();
		properties.setProperty("eroq.profile", "development");
		properties.setProperty("eroq.server.host", "127.0.0.1");
		properties.setProperty("eroq.server.port", "1980");
	}
	
	private void initProperties(String propertiesFileName) {
		if(propertiesFileName != null) {
			logger.debug("Looking in the classpath for a file named '"+propertiesFileName+"'");
			initProperties(Thread.currentThread().getContextClassLoader().getResource(propertiesFileName));
		} else {
			logger.debug("No file name provided");
		}
	}
	
	private void initProperties(URL propertiesFileURL) {
		InputStream propertiesInputStream = null;
		if(propertiesFileURL != null) {
			logger.debug("Found properties at "+propertiesFileURL.toString());
			try {
				propertiesInputStream = propertiesFileURL.openStream();
			} catch (IOException e) {
			}
			initProperties(propertiesInputStream);
		} else {
			logger.debug("Could not find properties");
		}
	}
	
	private void initProperties(InputStream propertiesInputStream) {
		if(propertiesInputStream != null) {
			logger.debug("Opened properties input stream");			
			try {
				properties.load(propertiesInputStream);
				logger.debug("Loaded properties successfully");
			} catch (IOException e) {
				throw new Error("Could not load properties", e);
			}
		} else {
			logger.debug("Could not open properties input stream");
		}
	}
	
	private void printCurrentProperties() {
		logger.debug("Active application properties:");
		for(Object key: properties.keySet()) {
			Object value = properties.get(key);
			logger.debug("\t{} = {}", key, value);
		}
	}
	
	public Properties getProperties() {
		return properties;
	}
	
	private void request(AppHttpContext ctx) throws Exception {
		// This gets run only if this is the main App being run
		run(ctx, () -> {
			// If we get here it means we couldn't find a defined resource
			// for the ctx.getMethdod()/ctx.getPath() pair so we will
			// just let the client know that the resource was not found
			ctx.setStatus(HttpStatus.NOT_FOUND);
			ctx.setBody("Cannot "+ctx.getMethod()+" "+ctx.getPath());
		});
	}
	
	public void listen(Integer port, String host, HttpServerListenCallback listenCallback) {
		HttpServer httpServer = Http.createServer((ctx) -> {
			AppHttpContext appCtx = new AppHttpContext(ctx, this);
			try {
				request(appCtx);
			} catch (Exception exception) {
				// Weird that we got here so log the exception
				//requestError(appCtx, exception);
			}
		});
		httpServer.listen(port, host, (server) -> {
			done(server);
			if(listenCallback != null) {
				listenCallback.call(server);				
			}
		});
	}
	
	public void listen(HttpServerListenCallback listenCallback) {
		String host = properties.getProperty("eroq.server.host");
		Integer port = Integer.valueOf(properties.getProperty("eroq.server.port"));
		listen(port, host, listenCallback);
	}

	public void listen(Integer port, String host) {
		listen(port, host, null);
	}

	public void listen() {
		listen(null);
	}
	
}
