package org.eroq.router;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eroq.pathmatcher.PathKey;
import org.eroq.pathmatcher.PathMatcher;

public class RouteMatcher {
	
	private static enum RouteMatcherType {
		PATH("path"),
		PATTERN("pattern");
		private String displayName;
		private RouteMatcherType(String displayName) {
			this.displayName = displayName;
		}
		public String getDisplayName() {
			return displayName;
		}
	}
	public static class Builder {
		private RouteMatcherType type;
		private String path;
		private Pattern pattern;
		private boolean sensitive;
		private boolean strict;
		private boolean end;
		private Builder(RouteMatcherType type) {
			this.type = type;
			this.path = null;
			this.pattern = null;
			this.sensitive = false;
			this.strict = false;
			this.end = true;
		}
		public static Builder createWithPath(String path) {
			Builder builder = new Builder(RouteMatcherType.PATH);
			builder.path = path;
			return builder;
		}
		public static Builder createWithPattern(Pattern pattern) {
			Builder builder = new Builder(RouteMatcherType.PATTERN);
			builder.pattern = pattern;
			return builder;
		}
		public Builder sensitive(boolean sensitive) {
			if(this.type == RouteMatcherType.PATTERN) {
				throw new IllegalArgumentException("This option cannot be applied to a builder of type "+this.type.getDisplayName());
			}
			this.sensitive = sensitive;
			return this;
		}
		public Builder strict(boolean strict) {
			if(this.type == RouteMatcherType.PATTERN) {
				throw new IllegalArgumentException("This option cannot be applied to a builder of type "+this.type.getDisplayName());
			}
			this.strict = strict;
			return this;
		}
		public Builder end(boolean end) {
			if(this.type == RouteMatcherType.PATTERN) {
				throw new IllegalArgumentException("This option cannot be applied to a builder of type "+this.type.getDisplayName());
			}
			this.end = end;
			return this;
		}
		public RouteMatcher build() {
			switch(this.type) {
			case PATH:
				return new RouteMatcher(this.path, this.sensitive, this.strict, this.end);
			case PATTERN:
				return new RouteMatcher(this.pattern);
			default:
				return null;
			}
		}
	}
	
	private PathMatcher pathMatcher;
	
	public PathMatcher getPathMatcher() {
		return pathMatcher;
	}
	
	private RouteMatcher(String path, boolean sensitive, boolean strict, boolean end) {
		pathMatcher = new PathMatcher(path, sensitive, strict, end);
	}

	private RouteMatcher(Pattern pattern) {
		pathMatcher = new PathMatcher(pattern);
	}
	
	public RouteMatch match(String path) {
		Matcher matcher = pathMatcher.getPattern().matcher(path);
		if(matcher.find()) {
			List<PathKey> keys = pathMatcher.getKeys();
			RouteParams params = new RouteParams();
			for(int i=0; i<matcher.groupCount(); i++) {
				Object keyName = keys.get(i).getName();
				String matchedValue = matcher.group(i+1);
				params.add(keyName, matchedValue);
			}
			return new RouteMatch(params, matcher);
		}
		return null;
	}
	
	@Override
	public String toString() {
		return pathMatcher.toString();
	}
	
	@Override
	public boolean equals(Object obj) {
		if(this == obj) {
			return true;
		}
		if(!(obj instanceof RouteMatcher)) {
			return false;
		}
		RouteMatcher routeObj = (RouteMatcher) obj;
		return routeObj.pathMatcher.equals(routeObj.pathMatcher);
	}

}
