package org.eroq.router;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.eroq.http.HttpVerb;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Router implements Middleware {
	
	private static Logger logger = LoggerFactory.getLogger(Router.class);
	
	/******************************************
	 * 
	 * Router properties methods
	 * 
	 ******************************************/
	
	protected Properties properties;
	
	public void setProperties(Properties properties) {
		this.properties = properties;
	}
	
	public Properties getProperties() {
		return properties;
	}
	
	/******************************************
	 * 
	 * Routing implementation
	 * 
	 ******************************************/

	@Override
	public void run(AppHttpContext ctx, AppNextMiddlewareHandler next) {
		if(usedMiddleware.size()==0) {
			next.handle();
			return;
		}
		runMiddleware(ctx, 0, next);
	}
	
	private void runMiddleware(AppHttpContext ctx, int middlewareIndex, AppNextMiddlewareHandler last) {
		if(middlewareIndex >= usedMiddleware.size()) {
			last.handle();
		}
		Middleware middleware = usedMiddleware.get(middlewareIndex);
		middleware.run(ctx, () -> {
			runMiddleware(ctx, middlewareIndex+1, last);
		});
	}
	
	
	/******************************************
	 * 
	 * Queue middleware in router
	 * 
	 ******************************************/
	
	private List<Middleware> usedMiddleware = new ArrayList<>();

	public void use(Middleware... middlewares) {
		usedMiddleware.addAll(Arrays.asList(middlewares));
	}
	
	/******************************************
	 * 
	 * Middleware for Router
	 * 
	 ******************************************/
	
	public Middleware router(String path, Middleware middleware) {
		RouteMatcher routeMatcher = RouteMatcher.Builder.createWithPath(path).end(false).build();
		logger.debug("ROUTER: {}", path);
		return (ctx, next) -> {
			RouteMatch routeMatch = routeMatcher.match(ctx.getPath());
			if(routeMatch != null) {
				logger.debug("Matched: ROUTER @ {} with params:", routeMatcher.getPathMatcher().getPattern());
				RouteParams routeParams = routeMatch.getParams();
				for(Object key: routeParams.keys()) {
					logger.debug("\t{}: {}", key, routeParams.getAll(key));
				}
				AppHttpContext tmpCtx = ctx.createTemporaryContext();
				tmpCtx.setAttribute("params", routeParams);

				// Call any param registered middleware
				processParams(tmpCtx, routeMatch.getParams());
				
				// Adjust baseUrl, path and originalUrl
				String currentPath = (String) tmpCtx.getAttribute("path");		
				String currentBaseUrl = (String) tmpCtx.getAttribute("baseUrl");
				if(currentBaseUrl == null) {
					currentBaseUrl = "";
				}
				String subBaseUrl = currentBaseUrl + currentPath.substring(0, routeMatch.getMatcher().end());
				if(subBaseUrl.endsWith("/")) {
					subBaseUrl = subBaseUrl.substring(0, subBaseUrl.length()-1);
				}
				String subPath = currentPath.substring(routeMatch.getMatcher().end());
				if(subPath.length()==0) {
					subPath = "/";
				}
				tmpCtx.setAttribute("baseUrl", subBaseUrl);
				tmpCtx.setAttribute("path", subPath);								
				tmpCtx.setAttribute("originalUrl", subBaseUrl + subPath);
				
				logger.debug("--> baseUrl={} path={} ", subBaseUrl, subPath);

				// TODO: Check wrapper structure of middleware here
				middleware.run(tmpCtx, next);
				
				logger.debug("<-- baseUrl={} path={} ", currentBaseUrl, currentPath);
				
			} else {
				next.handle();				
			}
		};
	}
	
	public void route(String path, Middleware middleware) {
		use(router(path, middleware));
	}

	/******************************************
	 * 
	 * Middleware for HTTP verbs
	 * 
	 ******************************************/

	public Middleware endpoint(String method, String path, EndpointMiddleware middleware) {
		RouteMatcher routeMatcher = RouteMatcher.Builder.createWithPath(path).build();
		logger.debug("ENPOINT: {} @ {}", (method!=null)?method:"ALL", path);
		return (ctx, next) -> {
			if(method==null || method.equalsIgnoreCase(ctx.getMethod())) {
				RouteMatch routeMatch = routeMatcher.match(ctx.getPath());
				if(routeMatch != null) {
					logger.debug("Matched: {} @ {} with params:", (method!=null?method:"ALL"), routeMatcher.getPathMatcher().getPattern());
					RouteParams routeParams = routeMatch.getParams();
					for(Object key: routeParams.keys()) {
						logger.debug("\t{}: {}", key, routeParams.getAll(key));
					}
					AppHttpContext tmpCtx = ctx.createTemporaryContext();
					tmpCtx.setAttribute("params", routeParams);
					// Call any param registered middleware
					processParams(tmpCtx, routeMatch.getParams());
					// Call middleware
					middleware.run(tmpCtx);						
					return;
				}
			}
			logger.trace("Did not match: {} @ {}", (method!=null?method:"ALL"), routeMatcher);
			next.handle();
		};
	}

	public void all(String path, EndpointMiddleware middleware) {
		use(endpoint(null, path, middleware));
	}
	public void get(String path, EndpointMiddleware middleware) {
		use(endpoint(HttpVerb.GET.value(), path, middleware));
	}
	public void post(String path, EndpointMiddleware middleware) {
		use(endpoint(HttpVerb.POST.value(), path, middleware));
	}
	public void put(String path, EndpointMiddleware middleware) {
		use(endpoint(HttpVerb.PUT.value(), path, middleware));
	}
	public void patch(String path, EndpointMiddleware middleware) {
		use(endpoint(HttpVerb.PATCH.value(), path, middleware));
	}
	public void delete(String path, EndpointMiddleware middleware) {
		use(endpoint(HttpVerb.DELETE.value(), path, middleware));
	}
	public void head(String path, EndpointMiddleware middleware) {
		use(endpoint(HttpVerb.HEAD.value(), path, middleware));
	}
	public void options(String path, EndpointMiddleware middleware) {
		use(endpoint(HttpVerb.OPTIONS.value(), path, middleware));
	}
	
	/******************************************
	 * 
	 * Middleware for params
	 * 
	 ******************************************/
	
	private List<String> registeredParameters = new ArrayList<>();
	private Map<String, List<ParamMiddleware>> registeredParametersMapping = new HashMap<>();
	
	public void param(String parameter, ParamMiddleware paramMiddleware) {
		List<ParamMiddleware> registeredParamMiddlewares = registeredParametersMapping.get(parameter);
		if(registeredParamMiddlewares == null) {
			registeredParamMiddlewares = new ArrayList<>();
			registeredParametersMapping.put(parameter, registeredParamMiddlewares);
			registeredParameters.add(parameter);
			logger.debug("PARAM: {}", parameter);
		}
		registeredParamMiddlewares.add(paramMiddleware);
	}
	
	private void runParamMiddleware(AppHttpContext ctx, RouteParams params, Object paramId, int paramMiddlewareIndex) {
		List<ParamMiddleware> registeredParametersForName = registeredParametersMapping.get(paramId);
		if(paramMiddlewareIndex >= registeredParametersForName.size()) {
			return;
		}
		ParamMiddleware paramMiddleware = registeredParametersForName.get(paramMiddlewareIndex);
		paramMiddleware.run(ctx, () -> {
			runParamMiddleware(ctx, params, paramId, paramMiddlewareIndex+1);
		}, params.getLast(paramId));
	}
	
	private void processParams(AppHttpContext ctx, RouteParams params) {
		
		// This will hold whether params have been handled in the request/response cycle
		@SuppressWarnings("unchecked")
		Set<Object> processedParametersMapping = (Set<Object>) ctx.getAttribute("_processedParametersMapping");
		if(processedParametersMapping == null) {
			processedParametersMapping = new HashSet<>();
			ctx.setAttribute("_processedParametersMapping", processedParametersMapping);
		}
		
		// For each detected parameter:
		//   - run its registered param middleware until next is not called anymore
		for(Object paramId: params.keys()) {
			if(!processedParametersMapping.contains(paramId)) {
				processedParametersMapping.add(paramId);
				logger.debug("Processing param: {}", paramId);
				runParamMiddleware(ctx, params, paramId, 0);			
			}
		}
		
	}



}
