package org.evrete.collections;

import org.evrete.util.ForkingMap;
import org.evrete.util.Indexed;
import org.evrete.util.MapEntryImpl;

import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

/**
 * Abstract array-backed data structure that provides fast access to its values using a generated key.
 * The generated key serves as a replacement for {@link Object#equals(Object)} or {@link Object#hashCode()}
 * used in traditional map-like structures. The purpose of this structure is to deliver <code>put</code> performance
 * similar to that of Java maps and <code>get</code> performance comparable to Java arrays.
 * <p>
 * The map has a "branching" constructor that can be used by {@link org.evrete.api.Copyable} implementations.
 * </p>
 *
 * @param <T>        the type for which the indexed store is created
 * @param <MatchKey> the type of the key by which values are compared
 * @param <FastKey>  the type of the generated key
 * @param <Stored>   the type of the stored values
 */
public abstract class ForkingArrayMap<T, MatchKey, FastKey extends Indexed, Stored> {
    private final ForkingArray<IndexedMapEntry<FastKey, Stored>> array;
    private final ForkingMap<MatchKey, IndexedMapEntry<FastKey, Stored>> keyMap;
    private final Function<T, MatchKey> keyFunction;

    protected ForkingArrayMap(Function<T, MatchKey> keyFunction) {
        this.keyFunction = keyFunction;
        this.keyMap = new ForkingMap<>();
        this.array = new ForkingArray<>();
    }

    /**
     * The "branching" constructor
     *
     * @param other the parent object from a new branch is created
     */
    protected ForkingArrayMap(ForkingArrayMap<T, MatchKey, FastKey, Stored> other) {
        this.keyFunction = other.keyFunction;
        this.keyMap = other.keyMap.nextBranch();
        this.array = other.array.newBranch();
    }

    public Stream<Stored> values() {
        return array.stream().map(MapEntryImpl::getValue);
    }

    /**
     * Generates a unique key for the given value. This key will be used for accessing
     * the value within the map.
     *
     * @param value the value for which the key is to be generated
     * @param index the index at which the value is to be stored
     * @return a generated key for the given value
     */
    protected abstract FastKey generateKey(T value, int index);

    /**
     * Generates the stored value using the provided key and original value. This
     * stored value is what will be stored in this map.
     *
     * @param key   the generated key for the value
     * @param value the original value to be stored
     * @return the generated stored value
     */
    protected abstract Stored generateValue(FastKey key, T value);

    /**
     * Updates saved values
     * @param operator the update function
     */
    public void update(UnaryOperator<Stored> operator) {
        array.update(entry -> new IndexedMapEntry<>(entry.getKey(), operator.apply(entry.getValue())));
    }

    public int size() {
        return array.size();
    }

    public void forEach(Consumer<Map.Entry<FastKey, Stored>> consumer) {
        this.array.forEach(consumer);
    }

    public void forEachValue(Consumer<Stored> consumer) {
        this.array.forEach(entry -> consumer.accept(entry.getValue()));
    }

    /**
     * Retrieves an existing entry or creates a new one if no entry exists for the given value.
     *
     * @param value the value for which the entry is to be retrieved or created
     * @return an entry containing the generated key and stored value for the given value
     */
    public synchronized Map.Entry<FastKey, Stored> getOrCreateEntry(T value) {
        MatchKey valueKey = keyFunction.apply(value);
        IndexedMapEntry<FastKey, Stored> found = keyMap.get(valueKey);
        if (found == null) {
            // Generating new key
            found = array.append(value, (i, v) -> {
                FastKey newKey = generateKey(value, i);
                Stored newValue = generateValue(newKey, value);
                return new IndexedMapEntry<>(newKey, newValue);
            });
            keyMap.put(valueKey, found);
        }
        return found;
    }

    /**
     * The getter for stored values.
     *
     * @param key the key that was previously generated by the {@link #getOrCreateEntry(Object)} method
     * @return stored value
     */
    public Stored get(FastKey key) {
        IndexedMapEntry<FastKey, Stored> entry = array.get(key.getIndex());
        return entry == null ? null : entry.getValue();
    }

    private static class IndexedMapEntry<K extends Indexed, V> extends MapEntryImpl<K, V> implements Indexed {

        public IndexedMapEntry(K key, V value) {
            super(key, value);
        }

        @Override
        public int getIndex() {
            return getKey().getIndex();
        }
    }

}
