/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH
 * under one or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information regarding copyright
 * ownership. Camunda licenses this file to you under the Apache License,
 * Version 2.0; you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.eximeebpms.bpm.engine.test.api.authorization.externaltask;

import static org.eximeebpms.bpm.engine.test.api.authorization.util.AuthorizationScenario.scenario;
import static org.eximeebpms.bpm.engine.test.api.authorization.util.AuthorizationSpec.grant;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eximeebpms.bpm.engine.authorization.BatchPermissions;
import org.eximeebpms.bpm.engine.authorization.Permissions;
import org.eximeebpms.bpm.engine.authorization.Resources;
import org.eximeebpms.bpm.engine.batch.Batch;
import org.eximeebpms.bpm.engine.batch.history.HistoricBatch;
import org.eximeebpms.bpm.engine.externaltask.ExternalTask;
import org.eximeebpms.bpm.engine.externaltask.ExternalTaskQuery;
import org.eximeebpms.bpm.engine.repository.ProcessDefinition;
import org.eximeebpms.bpm.engine.runtime.Job;
import org.eximeebpms.bpm.engine.runtime.ProcessInstance;
import org.eximeebpms.bpm.engine.test.ProcessEngineRule;
import org.eximeebpms.bpm.engine.test.api.authorization.util.AuthorizationScenario;
import org.eximeebpms.bpm.engine.test.api.authorization.util.AuthorizationTestRule;
import org.eximeebpms.bpm.engine.test.api.runtime.migration.models.ExternalTaskModels;
import org.eximeebpms.bpm.engine.test.util.ProcessEngineTestRule;
import org.eximeebpms.bpm.engine.test.util.ProvidedProcessEngineRule;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameter;
import org.junit.runners.Parameterized.Parameters;

@RunWith(Parameterized.class)
public class SetExternalTasksRetriesBatchAuthorizationTest {

  public ProcessEngineRule engineRule = new ProvidedProcessEngineRule();
  public AuthorizationTestRule authRule = new AuthorizationTestRule(engineRule);
  protected ProcessEngineTestRule testRule = new ProcessEngineTestRule(engineRule);


  @Rule
  public RuleChain chain = RuleChain.outerRule(engineRule).around(authRule).around(testRule);

  @Parameter
  public AuthorizationScenario scenario;

  @Parameters(name = "Scenario {index}")
  public static Collection<AuthorizationScenario[]> scenarios() {
    return AuthorizationTestRule.asParameters(
      scenario()
        .withAuthorizations(
            grant(Resources.PROCESS_DEFINITION, "processDefinition", "userId", Permissions.READ, Permissions.READ_INSTANCE))
        .failsDueToRequired(
            grant(Resources.BATCH, "batchId", "userId", Permissions.CREATE),
            grant(Resources.BATCH, "batchId", "userId", BatchPermissions.CREATE_BATCH_SET_EXTERNAL_TASK_RETRIES)),
      scenario()
        .withAuthorizations(
            grant(Resources.PROCESS_DEFINITION, "processDefinition", "userId", Permissions.READ, Permissions.READ_INSTANCE, Permissions.UPDATE_INSTANCE),
            grant(Resources.BATCH, "batchId", "userId", Permissions.CREATE))
        .succeeds(),
      scenario()
        .withAuthorizations(
            grant(Resources.PROCESS_DEFINITION, "processDefinition", "userId", Permissions.READ, Permissions.READ_INSTANCE, Permissions.UPDATE_INSTANCE),
            grant(Resources.BATCH, "batchId", "userId", BatchPermissions.CREATE_BATCH_SET_EXTERNAL_TASK_RETRIES))
        .succeeds(),
      scenario()
        .withAuthorizations(
          grant(Resources.PROCESS_DEFINITION, "processDefinition", "userId", Permissions.READ, Permissions.READ_INSTANCE),
          grant(Resources.BATCH, "batchId", "userId", Permissions.CREATE),
          grant(Resources.PROCESS_INSTANCE, "processInstance1", "userId", Permissions.UPDATE))
        .succeeds(),
      scenario()
        .withAuthorizations(
            grant(Resources.PROCESS_DEFINITION, "processDefinition", "userId", Permissions.READ, Permissions.READ_INSTANCE),
            grant(Resources.BATCH, "batchId", "userId", Permissions.CREATE))
        .failsDueToRequired(
            grant(Resources.PROCESS_DEFINITION, "processDefinition", "userId", Permissions.UPDATE_INSTANCE),
            grant(Resources.PROCESS_INSTANCE, "processInstance1", "userId", Permissions.UPDATE))
      );
  }

  @Before
  public void setUp() {
    authRule.createUserAndGroup("userId", "groupId");
  }

  @After
  public void tearDown() {
    authRule.deleteUsersAndGroups();
  }

  @After
  public void cleanBatch() {
    Batch batch = engineRule.getManagementService().createBatchQuery().singleResult();
    if (batch != null) {
      engineRule.getManagementService().deleteBatch(
          batch.getId(), true);
    }

    HistoricBatch historicBatch = engineRule.getHistoryService().createHistoricBatchQuery().singleResult();
    if (historicBatch != null) {
      engineRule.getHistoryService().deleteHistoricBatch(
          historicBatch.getId());
    }
  }

  @Test
  public void testSetRetriesAsync() {

    // given
    ProcessDefinition processDefinition = testRule.deployAndGetDefinition(ExternalTaskModels.ONE_EXTERNAL_TASK_PROCESS);
    ProcessInstance processInstance1 = engineRule.getRuntimeService().startProcessInstanceByKey("Process");
    List<ExternalTask> externalTasks = engineRule.getExternalTaskService().createExternalTaskQuery().list();

    ArrayList<String> externalTaskIds = new ArrayList<String>();

    for (ExternalTask task : externalTasks) {
      externalTaskIds.add(task.getId());
    }

    // when
    authRule
      .init(scenario)
      .withUser("userId")
      .bindResource("batchId", "*")
      .bindResource("processInstance1", processInstance1.getId())
      .bindResource("processDefinition", processDefinition.getKey())
      .start();

    Batch batch = engineRule.getExternalTaskService().setRetriesAsync(externalTaskIds, null, 5);
    if (batch != null) {
      executeSeedAndBatchJobs(batch);
    }

    // then
    if (authRule.assertScenario(scenario)) {
      externalTasks = engineRule.getExternalTaskService().createExternalTaskQuery().list();
      for ( ExternalTask task : externalTasks) {
      Assert.assertEquals(5, (int) task.getRetries());
      }
    }
  }

  @Test
  public void testSetRetriesWithQueryAsync() {

    // given
    ProcessDefinition processDefinition = testRule.deployAndGetDefinition(ExternalTaskModels.ONE_EXTERNAL_TASK_PROCESS);
    ProcessInstance processInstance1 = engineRule.getRuntimeService().startProcessInstanceByKey("Process");
    List<ExternalTask> externalTasks;

    ExternalTaskQuery externalTaskQuery = engineRule.getExternalTaskService().createExternalTaskQuery();

    // when
    authRule
      .init(scenario)
      .withUser("userId")
      .bindResource("batchId", "*")
      .bindResource("processInstance1", processInstance1.getId())
      .bindResource("processDefinition", processDefinition.getKey())
      .start();

    Batch batch = engineRule.getExternalTaskService().setRetriesAsync(null, externalTaskQuery, 5);
    if (batch != null) {
      executeSeedAndBatchJobs(batch);
    }

    // then
    if (authRule.assertScenario(scenario)) {
      Assert.assertEquals("userId", batch.getCreateUserId());

      externalTasks = engineRule.getExternalTaskService().createExternalTaskQuery().list();
      for ( ExternalTask task : externalTasks) {
        Assert.assertEquals(5, (int) task.getRetries());
      }
    }
  }

  public void executeSeedAndBatchJobs(Batch batch) {
    Job job = engineRule.getManagementService().createJobQuery().jobDefinitionId(batch.getSeedJobDefinitionId()).singleResult();
    // seed job
    engineRule.getManagementService().executeJob(job.getId());

    for (Job pending : engineRule.getManagementService().createJobQuery().jobDefinitionId(batch.getBatchJobDefinitionId()).list()) {
      engineRule.getManagementService().executeJob(pending.getId());
    }
  }
}
