package org.exploit.blockbook;

import org.exploit.blockbook.model.transaction.BitcoinConfirmedTransaction;
import org.exploit.blockbook.model.transaction.BitcoinUnconfirmedTransaction;
import org.exploit.blockbook.model.transaction.BlockBookTransaction;
import org.exploit.blockbook.model.transaction.EvmTransaction;
import org.exploit.finja.core.ValueConverter;
import org.exploit.finja.core.constant.Asset;
import org.exploit.finja.core.constant.TxType;
import org.exploit.finja.core.converter.MathConstantConverter;
import org.exploit.finja.core.event.TxnEvent;
import org.exploit.finja.core.model.Value;

import java.math.BigInteger;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

public class BlockBookEventTransformer {
    private final Asset asset;
    private final ValueConverter valueConverter;

    public BlockBookEventTransformer(Asset asset, ValueConverter valueConverter) {
        this.asset = asset;
        this.valueConverter = valueConverter;
    }

    public List<TxnEvent> transform(String address, BlockBookTransaction tx) {
        if (tx instanceof BitcoinUnconfirmedTransaction) {
            return transformBitcoin(address, tx);
        } else if (tx instanceof BitcoinConfirmedTransaction) {
            return transformBitcoin(address, tx);
        } else if (tx instanceof EvmTransaction evm) {
            return transformEthereum(address, evm);
        } else {
            throw new IllegalArgumentException("Unknown transaction type: " + tx.getClass());
        }
    }

    private List<TxnEvent> transformBitcoin(String address, BlockBookTransaction tx) {
        return Optional.ofNullable(receiveEvent(address, tx))
                .map(Collections::singletonList)
                .orElseGet(() -> Optional.ofNullable(sendEvent(address, tx))
                        .map(Collections::singletonList)
                        .orElse(Collections.emptyList()));
    }

    private TxnEvent receiveEvent(String address, BlockBookTransaction tx) {
        var selfInputs = inputSum(address, tx);

        if (selfInputs != 0)
            return null;

        var sum = outputSum(address, tx);

        if (sum == 0)
            return null;

        return createTxnEvent(TxType.RECEIVE, address, tx, valueOf(sum));
    }

    private TxnEvent sendEvent(String address, BlockBookTransaction tx) {
        var sent = inputSum(address, tx);

        if (sent == 0)
            return null;

        var change = outputSum(address, tx);
        var dif = sent - change;

        if (dif <= 0)
            return null;

        return createTxnEvent(TxType.SEND, address, tx, valueOf(dif));
    }

    private long inputSum(String address, BlockBookTransaction tx) {
        return tx.getVin().stream()
                .filter(vin -> vin.getAddresses().contains(address))
                .mapToLong(vin -> Long.parseLong(vin.getValue()))
                .sum();
    }

    private long outputSum(String address, BlockBookTransaction tx) {
        return tx.getVout().stream()
                .filter(vout -> vout.getAddresses().contains(address))
                .mapToLong(vout -> Long.parseLong(vout.getValue()))
                .sum();
    }

    private Value valueOf(long unit) {
        var unitInt = BigInteger.valueOf(unit);
        var converted = valueConverter.toHuman(unitInt);
        return new Value(converted, unitInt);
    }

    private TxnEvent createTxnEvent(TxType type, String address, BlockBookTransaction tx, Value value) {
        return TxnEvent.builder()
                .type(type)
                .asset(asset)
                .address(address)
                .value(value)
                .txid(tx.getTxid())
                .confirmations(tx.getConfirmations())
                .timestamp(tx.getBlockTime())
                .build();
    }

    private List<TxnEvent> transformEthereum(String address, EvmTransaction tx) {
        var tokenTransfers = tx.getTokenTransfers();

        if (tokenTransfers == null || tokenTransfers.isEmpty())
            return processEthTransfer(address, tx);

        return processTokenTransfers(address, tx);
    }

    private List<TxnEvent> processEthTransfer(String address, EvmTransaction tx) {
        var input = tx.getVin().get(0);
        var output = tx.getVout().get(0);

        var from = input.getAddresses().get(0);
        var to = output.getAddresses().get(0);

        var type = from.equalsIgnoreCase(address) ? TxType.SEND : to.equalsIgnoreCase(address) ? TxType.RECEIVE : null;

        if (type == null)
            return Collections.emptyList();

        var unit = new BigInteger(output.getValue());
        var humanAmount = valueConverter.toHuman(unit);

        var timestamp = Instant.ofEpochSecond(tx.getBlockTime()).toEpochMilli();

        var event = TxnEvent.builder()
                .asset(asset)
                .txid(tx.getTxid())
                .value(new Value(humanAmount, unit))
                .type(type)
                .address(address)
                .timestamp(timestamp)
                .confirmations(tx.getConfirmations())
                .build();

        return List.of(event);
    }

    private List<TxnEvent> processTokenTransfers(String address, EvmTransaction tx) {
        var transfers = tx.getTokenTransfers();

        return transfers.stream()
                .map(transfer -> createEventFromTransfer(address, tx, transfer))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private TxnEvent createEventFromTransfer(String address, EvmTransaction tx, EvmTransaction.TokenTransfer transfer) {
        var from = transfer.getFrom();
        var to = transfer.getTo();

        var amount = new BigInteger(transfer.getValue());
        var humanAmount = new MathConstantConverter(transfer.getDecimals()).toHuman(amount);

        var txType = typeOf(from, to, address);

        if (txType == null)
            return null;

        return TxnEvent.builder()
                .address(address)
                .timestamp(timestamp(tx.getBlockTime()))
                .txid(tx.getTxid())
                .confirmations(tx.getConfirmations())
                .smartContract(transfer.getContract())
                .value(new Value(humanAmount, amount))
                .asset(asset)
                .build();
    }

    private TxType typeOf(String from, String to, String address) {
        return from.equalsIgnoreCase(address) ? TxType.SEND : to.equalsIgnoreCase(address) ? TxType.RECEIVE : null;
    }

    private long timestamp(long seconds) {
        return Instant.ofEpochSecond(seconds).toEpochMilli();
    }
}